<?php

namespace Modules\Qbo\Entities;

use App\Abstractions\Integrations\Accounting\AbstractAccountingAccount;
use App\Data\CreateNominalCodeData;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use App\Models\NominalCode;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Kirschbaum\PowerJoins\PowerJoins;
use Modules\Qbo\Database\factories\QboAccountFactory;
use Modules\Qbo\Enums\QboAccountTypeEnum;

/**
 * @property int $id
 * @property string $QboId
 * @property string $Name
 * @property string $FullyQualifiedName
 * @property QboAccountTypeEnum $AccountType
 * @property string $AccountSubType
 * @property string $Active
 * @property string $Classification
 * @property string $LastUpdatedTime
 * @property Carbon $LastUpdatedTimeUtc
 * @property array $json_object
 * @property-read NominalCode $nominalCode
 * @property Carbon $updated_at
 * @property Carbon $created_at
 */
class QboAccount extends QboModel implements AbstractAccountingAccount, Filterable
{
    use BulkImport;
    use HasFactory;
    use HasFilters;
    use PowerJoins;

    const BULK_THRESHOLD = 1;

    protected $guarded = [];

    protected $casts = [
        'AccountType' => QboAccountTypeEnum::class,
        'json_object' => 'array',
    ];

    public static function getUniqueField(): string
    {
        return 'QboId';
    }

    public function getNominalCodeData(): CreateNominalCodeData
    {
        return CreateNominalCodeData::from([
            'code' => $this->QboId,
            'name' => $this->Name,
            'type' => $this->AccountType->toNominalCodeType(),
        ]);
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function nominalCode(): HasOne
    {
        return $this->hasOne(NominalCode::class, 'code', 'QboId');
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    public static function newFactory(): Factory
    {
        return QboAccountFactory::new();
    }
}
