<?php

namespace Modules\Qbo\ApiParameterObjects;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Repositories\Accounting\AccountingTransactionRepository;
use Illuminate\Support\LazyCollection;
use Modules\Qbo\Data\QboCreatePurchaseOrderData;
use Modules\Qbo\Data\QboCreatePurchaseOrderLineData;
use Modules\Qbo\Entities\QboItem;
use Modules\Qbo\Repositories\QboAccountRepository;
use Modules\Qbo\Repositories\QboItemRepository;
use Modules\Qbo\Repositories\QboVendorRepository;

class QboUpdateOrCreatePurchaseOrdersApo implements ApiDataTransformerInterface
{
    public QboAccountRepository $accounts;

    public QboVendorRepository $vendors;

    public QboItemRepository $items;

    public AccountingTransactionRepository $accountingTransactionRepository;

    public function __construct(
        public LazyCollection $accountingTransactions
    ) {
        $this->accounts = app(QboAccountRepository::class);
        $this->vendors = app(QboVendorRepository::class);
        $this->items = app(QboItemRepository::class);
        $this->accountingTransactionRepository = app(AccountingTransactionRepository::class);
    }

    /**
     * Ideally this logic should go in manager but its moved here as Queue do not allow to accept closure.
     */
    private function sanitizePurchaseOrdersHavingMultipleTaxRates()
    {
        return $this->accountingTransactions->map(function (AccountingTransaction $accountingTransaction) {
            if ($accountingTransaction->is_tax_included) {
                $purchaseOrder = $accountingTransaction->link;
                if (is_null($purchaseOrder->tax_rate_id)) {
                    if ($purchaseOrder->salesOrderLines->pluck('tax_rate_id')->values()->unique()->count() > 1) {
                        $this->accountingTransactionRepository->updateLastSyncError($accountingTransaction, [
                            'Tax rate ids mis-match for order lines, it should be uniform for syncing with QuickBooks',
                        ]);

                        return null;
                    }
                }
            }

            return $accountingTransaction;
        })
            ->filter();
    }

    public function transform(): array
    {
        $purchaseOrders = [];

        $addSeparateTaxLine = true; //TODO: Optimize

        if (! $addSeparateTaxLine) {
            $transactions = $this->sanitizePurchaseOrdersHavingMultipleTaxRates();
        } else {
            $transactions = $this->accountingTransactions;
        }

        $transactions->each(function (AccountingTransaction $accountingTransaction) use (&$purchaseOrders, $addSeparateTaxLine) {
            $lineItems = [];
            $salesTax = 0;

            $accountingTransaction->accountingTransactionLines->each(function (AccountingTransactionLine $accountingTransactionLine) use (&$purchaseOrders, $accountingTransaction, &$lineItems, $addSeparateTaxLine, &$salesTax) {
                $amount = $addSeparateTaxLine ? $accountingTransactionLine->getAmountWithoutTax($accountingTransaction->is_tax_included) : $accountingTransactionLine->getAmountWithTax($accountingTransaction->is_tax_included);

                $itemName = QboItem::MISSING_SKU;

                if ($accountingTransactionLine->link->product?->sku) {
                    $itemName = $accountingTransactionLine->link->product?->sku;
                }

                $lineItems[] = QboCreatePurchaseOrderLineData::from([
                    'Description' => $accountingTransactionLine->description,
                    'DetailType' => 'ItemBasedExpenseLineDetail',
                    'Amount' => $amount * $accountingTransactionLine->quantity,
                    'ItemBasedExpenseLineDetail' => [
                        'ItemRef' => [
                            'value' => $this->items->getQboProductByName($itemName)->QboId,
                        ],
                        'Qty' => $accountingTransactionLine->quantity,
                        'UnitPrice' => $amount,
                        'TaxCodeRef' => [
                            'value' => $addSeparateTaxLine ? 'NON' : 'Tax',
                        ],
                    ],
                ])->toArray();

                if ($addSeparateTaxLine) {
                    $salesTax = $salesTax + $accountingTransactionLine->getTaxAmount($accountingTransaction->is_tax_included);
                }
            });

            //Pick tax rate for first sales order line items
            if ($addSeparateTaxLine) {
                $lineItems[] = QboCreatePurchaseOrderLineData::from([
                    'Description' => 'Sales Tax',
                    'DetailType' => 'ItemBasedExpenseLineDetail',
                    'Amount' => $salesTax,
                    'ItemBasedExpenseLineDetail' => [
                        'Qty' => 1,
                        'UnitPrice' => $salesTax,
                        'ItemRef' => [
                            'value' => $this->items->getQboProductByName(QboItem::TAX_PAYABLE_ITEM)->QboId,
                        ],
                    ],
                ])->toArray();
            }

            $data = QboCreatePurchaseOrderData::from([
                'TxnDate' => $accountingTransaction->link->purchase_order_date->format('Y-m-d'),
                'DocNumber' => $accountingTransaction->link->purchase_order_number,
                'VendorRef' => [
                    'value' => $this->vendors->getQboVendorByDisplayName($accountingTransaction->link->supplier->name)->QboId,
                ],
                'CurrencyRef' => [
                    'value' => $accountingTransaction->currency_code,
                ],
                'Line' => $lineItems,
            ]);

            if ($QboId = $accountingTransaction->accountingIntegration?->QboId) {
                $data->Id = $QboId;
                $data->SyncToken = $accountingTransaction->accountingIntegration->json_object['SyncToken'];
            }

            $purchaseOrders[] = $data->toArray();
        });

        $params = [
            'records' => $purchaseOrders,
        ];

        return array_filter($params, fn ($param) => ! is_null($param));
    }
}
