<?php

namespace Modules\Qbo\ApiParameterObjects;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Enums\AccountingTransactionLineTypeEnum;
use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use Illuminate\Support\LazyCollection;
use Modules\Qbo\Data\QboCreateJournalsData;
use Modules\Qbo\Data\QboCreateJournalsLineData;
use Modules\Qbo\Repositories\QboAccountRepository;
use Modules\Qbo\Repositories\QboCustomerRepository;
use Modules\Qbo\Repositories\QboItemRepository;

class QboUpdateOrCreateJournalsApo implements ApiDataTransformerInterface
{
    public QboAccountRepository $accounts;

    public QboCustomerRepository $customers;

    public QboItemRepository $items;

    const POSTING_TYPE_DEBIT = 'Debit';

    const POSTING_TYPE_CREDIT = 'Credit';

    public function __construct(
        public LazyCollection $accountingTransactions
    ) {
        $this->accounts = app(QboAccountRepository::class);
        $this->customers = app(QboCustomerRepository::class);
        $this->items = app(QboItemRepository::class);
    }

    private function createLineData(AccountingTransactionLine $accountingTransactionLine, float $amount, $postingType): QboCreateJournalsLineData
    {
        return QboCreateJournalsLineData::from([
            'Description' => $accountingTransactionLine->description,
            'DetailType' => 'JournalEntryLineDetail',
            'Amount' => $amount,
            'JournalEntryLineDetail' => [
                'PostingType' => $postingType,
                'AccountRef' => [
                    'value' => $accountingTransactionLine->nominalCode->code,
                ],
            ],
        ]);
    }

    public function transform(): array
    {
        $journals = [];

        $this->accountingTransactions->each(function (AccountingTransaction $accountingTransaction) use (&$journals) {
            $lineItems = [];

            $accountingTransaction->accountingTransactionLines->each(function (AccountingTransactionLine $accountingTransactionLine) use (&$journals, &$lineItems) {
                if ($accountingTransactionLine->type == AccountingTransactionLineTypeEnum::CREDIT->value) {
                    $lineItems[] = $this->createLineData($accountingTransactionLine, $accountingTransactionLine->amount * $accountingTransactionLine->quantity, self::POSTING_TYPE_CREDIT)->toArray();
                } else {
                    $lineItems[] = $this->createLineData($accountingTransactionLine, $accountingTransactionLine->amount * $accountingTransactionLine->quantity, self::POSTING_TYPE_DEBIT)->toArray();
                }
            });

            $data = QboCreateJournalsData::from([
                'DocNumber' => $accountingTransaction->id,
                'TxnDate' => $accountingTransaction->transaction_date->format('Y-m-d'),
                'Line' => $lineItems,
            ]);

            if ($QboId = $accountingTransaction->accountingIntegration?->QboId) {
                $data->Id = $QboId;
                $data->SyncToken = $accountingTransaction->accountingIntegration->json_object['SyncToken'];
            }

            $journals[] = $data->toArray();
        });

        $params = [
            'records' => $journals,
        ];

        return array_filter($params, fn ($param) => ! is_null($param));
    }
}
