<?php

namespace Modules\Qbo\ApiParameterObjects;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Models\FinancialLine;
use App\Models\SalesCredit;
use App\Models\SalesOrder;
use App\Repositories\Accounting\AccountingTransactionRepository;
use Illuminate\Support\LazyCollection;
use Modules\Qbo\Data\QboCreateInvoicesData;
use Modules\Qbo\Data\QboCreateInvoicesLineData;
use Modules\Qbo\Entities\QboCustomer;
use Modules\Qbo\Entities\QboItem;
use Modules\Qbo\Repositories\QboAccountRepository;
use Modules\Qbo\Repositories\QboCustomerRepository;
use Modules\Qbo\Repositories\QboItemRepository;

class QboUpdateOrCreateInvoicesApo implements ApiDataTransformerInterface
{
    public QboAccountRepository $accounts;

    public QboCustomerRepository $customers;

    public QboItemRepository $items;

    public AccountingTransactionRepository $accountingTransactionRepository;

    public function __construct(
        public LazyCollection $accountingTransactions
    ) {
        $this->accounts = app(QboAccountRepository::class);
        $this->customers = app(QboCustomerRepository::class);
        $this->items = app(QboItemRepository::class);
        $this->accountingTransactionRepository = app(AccountingTransactionRepository::class);
    }

    /**
     * Ideally this logic should go in manager but its moved here as Queue do not allow to accept closure.
     */
    private function sanitizeSalesOrdersHavingMultipleTaxRates()
    {
        return $this->accountingTransactions->map(function (AccountingTransaction $accountingTransaction) {
            if ($accountingTransaction->is_tax_included) {
                $salesOrder = $accountingTransaction->link;
                if (is_null($salesOrder->tax_rate_id)) {
                    if ($salesOrder->salesOrderLines->pluck('tax_rate_id')->values()->unique()->count() > 1) {
                        $this->accountingTransactionRepository->updateLastSyncError($accountingTransaction, [
                            'Tax rate ids mis-match for order lines, it should be uniform for syncing with QuickBooks',
                        ]);

                        return null;
                    }
                }
            }

            return $accountingTransaction;
        })
            ->filter();
    }

    public function transform(): array
    {
        $invoices = [];

        $addSeparateTaxLine = true; //TODO: Optimize

        if (! $addSeparateTaxLine) {
            $transactions = $this->sanitizeSalesOrdersHavingMultipleTaxRates();
        } else {
            $transactions = $this->accountingTransactions;
        }

        $transactions->each(function (AccountingTransaction $accountingTransaction) use (&$invoices, $addSeparateTaxLine) {
            $lineItems = [];
            $salesTax = 0;

            $accountingTransaction->accountingTransactionLines->each(function (AccountingTransactionLine $accountingTransactionLine) use (&$invoices, $accountingTransaction, &$lineItems, $addSeparateTaxLine, &$salesTax) {
                $amount = $addSeparateTaxLine ? $accountingTransactionLine->getAmountWithoutTax($accountingTransaction->is_tax_included) : $accountingTransactionLine->getAmountWithTax($accountingTransaction->is_tax_included);
                $itemName = QboItem::MISSING_SKU;

                if ($accountingTransactionLine->link->product?->sku) {
                    $itemName = $accountingTransactionLine->link->product?->sku;
                } elseif ($accountingTransactionLine->link_type == FinancialLine::class) {
                    $itemName = $accountingTransactionLine->description;
                }

                // Look up specs of posting purchase orders... create a dto and build it here or get from a model?
                $lineItems[] = QboCreateInvoicesLineData::from([
                    'Description' => $accountingTransactionLine->description,
                    'DetailType' => 'SalesItemLineDetail',
                    'Amount' => $amount * $accountingTransactionLine->quantity,
                    'SalesItemLineDetail' => [
                        'ItemRef' => [
                            'value' => $this->items->getQboProductByName($itemName)->QboId,
                        ],
                        'Qty' => $accountingTransactionLine->quantity,
                        'UnitPrice' => $amount,
                        'TaxCodeRef' => [
                            'value' => $addSeparateTaxLine ? 'NON' : 'Tax',
                        ],
                    ],

                ])->toArray();

                if ($addSeparateTaxLine) {
                    $salesTax = $salesTax + $accountingTransactionLine->getTaxAmount($accountingTransaction->is_tax_included);
                }
            });

            //Pick tax rate for first sales order line items
            $taxCode = null;
            if (! $addSeparateTaxLine) {
                $taxCode = null;
                if (! @$accountingTransaction->link?->taxRate?->accountingIntegration) {
                    $taxCode = $accountingTransaction->link->salesOrderLines->whereNotNull('tax_rate_id')->first()?->taxRate?->accountingIntegration;
                } else {
                    $taxCode = $accountingTransaction->link?->taxRate?->accountingIntegration;
                }
            } else {
                $lineItems[] = QboCreateInvoicesLineData::from([
                    'Description' => 'Sales Tax',
                    'DetailType' => 'SalesItemLineDetail',
                    'Amount' => $salesTax,
                    'SalesItemLineDetail' => [
                        'ItemRef' => [
                            'value' => $this->items->getQboProductByName(QboItem::TAX_PAYABLE_ITEM)->QboId,
                        ],
                        'Qty' => 1,
                        'UnitPrice' => $salesTax,
                    ],

                ])->toArray();
            }

            if (get_class($accountingTransaction->link) == SalesOrder::class) {
                $docNumber = $accountingTransaction->link->sales_order_number;
            } elseif (get_class($accountingTransaction->link) == SalesCredit::class) {
                $docNumber = $accountingTransaction->link->sales_credit_number;
            }

            $data = QboCreateInvoicesData::from([
                'TxnDate' => $accountingTransaction->link->order_date->format('Y-m-d'),
                'DocNumber' => $docNumber,
                'CustomerRef' => [
                    'value' => $accountingTransaction->link->customer?->name ? $this->customers->getQboCustomerByDisplayName($accountingTransaction->link->customer->name)?->QboId : $this->customers->getQboCustomerByDisplayName(QboCustomer::MISSING_CUSTOMER_NAME)->QboId,
                ],
                'CurrencyRef' => [
                    'value' => $accountingTransaction->currency_code,
                ],
                'Line' => $lineItems,
                'TxnTaxDetail' => [
                    'TxnTaxCodeRef' => [
                        'value' => $taxCode?->QboId,
                    ],
                ],
            ]);

            if ($QboId = $accountingTransaction->accountingIntegration?->QboId) {
                $data->Id = $QboId;
                $data->SyncToken = $accountingTransaction->accountingIntegration->json_object['SyncToken'];
            }

            $invoices[] = $data->toArray();
        });

        $params = [
            'records' => $invoices,
        ];

        return array_filter($params, fn ($param) => ! is_null($param));
    }
}
