<?php

declare(strict_types=1);

namespace Modules\Ebay\ValueObject\Oauth;

use Stringable;

/**
 * Class State
 */
class State implements \ArrayAccess, Stringable
{
    /**
     * The delimiter used to separate the elements of the state
     */
    public const ELEMENT_DELIMITER = '_';

    /**
     * The delimiter used to separate the key and value of the state
     */
    public const PAIR_DELIMITER = '=';

    /**
     * @var array<int, mixed>
     */
    protected array $state;

    public function __construct(array $state)
    {
        $this->state = $state;
    }

    public static function fromArray(array $values): static
    {
        return new static($values);
    }

    public static function fromString(string $state): static
    {
        $decoded = base64_decode($state);
        $stateTuples = explode(static::ELEMENT_DELIMITER, $decoded);

        $parsedState = [];
        foreach ($stateTuples as $stateTuple) {
            $parts = explode(static::PAIR_DELIMITER, $stateTuple);
            $parsedState[$parts[0]] = $parts[1];
        }

        return new static($parsedState);
    }

    /**
     * Useful when comparing a returned State from the OAuth server to the one we sent
     * to the OAuth server during the first stage of the OAuth handshake.
     */
    public function isEqualTo(State $anotherStateObject): bool
    {
        return $this->toString() === $anotherStateObject->toString();
    }

    public function toString(): string
    {
        $condensedTuples = [];
        foreach ($this->state as $key => $value) {
            $condensedTuples[] = $key.static::PAIR_DELIMITER.$value;
        }

        return base64_encode(implode(static::ELEMENT_DELIMITER, $condensedTuples));
    }

    public function __toString(): string
    {
        return $this->toString();
    }

    public function offsetExists(mixed $offset): bool
    {
        return isset($this->state[$offset]);
    }

    public function offsetGet(mixed $offset): mixed
    {
        return $this->state[$offset];
    }

    public function offsetSet(mixed $offset, mixed $value): void
    {
        $this->state[$offset] = $value;
    }

    public function offsetUnset(mixed $offset): void
    {
        unset($this->state[$offset]);
    }
}
