<?php

namespace Feature\Managers;

use App\Exceptions\SalesOrder\SalesOrderFulfillmentDispatchException;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesCredit;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Warehouse;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Exception;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Facades\Queue;
use Modules\Ebay\Data\EbayOrderData;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\Ebay\Entities\EbayOrder;
use Modules\Ebay\Entities\EbayProduct;
use Modules\Ebay\Jobs\EbayFulfillmentJob;
use Modules\Ebay\Managers\EbayOrderManager;
use Modules\Ebay\Repositories\EbayOrderRepository;
use Modules\Ebay\Tests\EbayMockRequests;
use Modules\Ebay\Tests\EbayTestingData;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class EbayOrderManagerTest extends TestCase
{
    use FastRefreshDatabase;
    use EbayMockRequests;

    private EbayIntegrationInstance $ebayIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();
        $this->ebayIntegrationInstance = EbayIntegrationInstance::factory([
                'is_automatic_sync_enabled' => true,
            ])
            ->has(SalesChannel::factory())
            ->create();

        Queue::fake();
    }

    /**
     * @throws SalesOrderFulfillmentDispatchException
     * @throws BindingResolutionException
     * @throws Throwable
     */
    public function test_ebay_fulfillment_gets_submitted_on_fulfill(): void
    {
        $warehouse = Warehouse::first();
        $product = Product::factory()->create();

        $product->setInitialInventory($warehouse->id, 100, 5.00);

        $ebayOrder = EbayOrder::factory([
            'integration_instance_id' => $this->ebayIntegrationInstance->id
        ])
            ->hasOrderItems()
            ->create();

        $salesOrder = SalesOrder::factory([
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'sales_order_number' => $ebayOrder->orderId,
            'sales_channel_order_id' => $ebayOrder->id,
            'sales_channel_order_type' => EbayOrder::class
        ])
            ->has(SalesOrderLine::factory()->state([
                'sales_channel_line_id' => $ebayOrder->orderItems->first()->lineItemId,
                'product_id' => $product->id,
                'warehouse_id' => $warehouse->id,
            ]))
            ->create();

        $fulfillmentDate = now();

        app(FulfillmentManager::class)->fulfill($salesOrder, [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'warehouse_id' => $warehouse->id,
            'fulfilled_at' => $fulfillmentDate,
            'fulfilled_shipping_method' => 'USPS',
            'tracking_number' => '123456789',
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrder->salesOrderLines->first()->id,
                    'quantity' => 1,
                ]
            ],
        ], false);

        Queue::assertPushed(EbayFulfillmentJob::class);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_ebay_order_cancellation_cancels_sales_order(): void
    {
        $manager = new EbayOrderManager($this->ebayIntegrationInstance);

        $orderData = EbayOrderData::collection([EbayTestingData::getEbayCanceledOrderStage1()])->toCollection();
        app(EbayOrderRepository::class)->save($this->ebayIntegrationInstance, $orderData);

        $ebayProduct = EbayLegacyProduct::factory([
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
            'json_object' => [
                'ItemID' => '170468028584',
                'SKU' => 'ORD_VDCCHEVYCAM10',
            ],
        ])->create()->refresh();

        $product = Product::factory([
            'sku' => 'VDCCHEVYCAM10',
        ])->create();

        ProductListing::factory([
            'product_id' => $product->id,
            'listing_sku' => 'ORD_VDCCHEVYCAM10',
            'document_id' => $ebayProduct->id,
            'document_type' => EbayLegacyProduct::class,
        ])->create();

        $ebayOrder = EbayOrder::first();

        $manager->createSkuOrders([$ebayOrder->id]);

        $orderData = EbayOrderData::collection([EbayTestingData::getEbayCanceledOrderStage3()])->toCollection();
        app(EbayOrderRepository::class)->save($this->ebayIntegrationInstance, $orderData);

        $manager->handleCancelSalesOrders(collect());

        $this->assertDatabaseHas('sales_orders', [
            'id' => $ebayOrder->salesOrder->id,
            'order_status' => SalesOrder::STATUS_CLOSED,
        ]);

        $this->assertNotNull($ebayOrder->salesOrder->canceled_at);
    }
}