<?php

namespace Feature\Managers;

use App\Exceptions\SalesChannelProductMissingSkuException;
use App\Models\DataImportMapping;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\Warehouse;
use Carbon\Carbon;
use Exception;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\Ebay\Entities\EbayProductSetting;
use Modules\Ebay\Managers\EbayLegacyProductManager;
use Modules\Ebay\Tests\EbayMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class EbayLegacyProductManagerTest extends TestCase
{
    use FastRefreshDatabase;
    use EbayMockRequests;

    private EbayIntegrationInstance $ebayIntegrationInstance;
    private EbayLegacyProductManager $manager;

    /**
     * @throws Exception
     */
    public function setUp(): void
    {
        parent::setUp();
        $this->ebayIntegrationInstance = EbayIntegrationInstance::factory([
            'is_automatic_sync_enabled' => true,
        ])
            ->has(SalesChannel::factory())
            ->create();

        $this->manager = new EbayLegacyProductManager($this->ebayIntegrationInstance);

        $this->mockGetAccessToken();
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_relist_ended_ebay_auctions()
    {
        $this->mockRelistItem();
        $this->mockGetItem();

        $product = Product::factory()->create();

        $product->setInitialInventory(Warehouse::first()->id, 100, 5.00);

        $ebayProductEndedEarly = EbayLegacyProduct::factory()->create([
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
        ]);

        $jsonObject = $ebayProductEndedEarly->json_object;
        $jsonObject['ListingDuration'] = 'Days_7';
        $jsonObject['ListingType'] = 'Chinese';
        $jsonObject['SellingStatus']['CurrentPrice']['value'] = 100.00;
        $jsonObject['SellingStatus']['CurrentPrice']['currencyID'] = 'USD';
        $jsonObject['EndTimeUtc'] = Carbon::now()->subDay()->toDateTimeString();
        $jsonObject['ListingDetails']['EndingReason'] = 'NotAvailable';
        $jsonObject['SellingStatus']['BidCount'] = 0;
        $jsonObject['SellingStatus']['ListingStatus'] = 'Completed';
        $ebayProductEndedEarly->json_object = $jsonObject;
        $ebayProductEndedEarly->save();
        $ebayProductEndedEarly->refresh();

        $ebayProductSold = EbayLegacyProduct::factory()->create([
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
        ]);

        $jsonObject = $ebayProductSold->json_object;
        $jsonObject['ListingDuration'] = 'Days_7';
        $jsonObject['ListingType'] = 'Chinese';
        $jsonObject['SellingStatus']['CurrentPrice']['value'] = 100.00;
        $jsonObject['SellingStatus']['CurrentPrice']['currencyID'] = 'USD';
        $jsonObject['EndTimeUtc'] = Carbon::now()->subDay()->toDateTimeString();
        $jsonObject['SellingStatus']['BidCount'] = 1;
        $jsonObject['SellingStatus']['ListingStatus'] = 'Completed';
        $ebayProductSold->json_object = $jsonObject;
        $ebayProductSold->save();
        $ebayProductSold->refresh();

        $ebayProductUnsold = EbayLegacyProduct::factory()->create([
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
        ]);

        $jsonObject = $ebayProductUnsold->json_object;
        $jsonObject['ItemID'] = '123';
        $jsonObject['SKU'] = 'SKU-RELIST';
        $jsonObject['ListingDuration'] = 'Days_7';
        $jsonObject['ListingType'] = 'Chinese';
        $jsonObject['Site'] = 'US';
        $jsonObject['SellingStatus']['CurrentPrice']['value'] = 100.00;
        $jsonObject['SellingStatus']['CurrentPrice']['currencyID'] = 'USD';
        $jsonObject['EndTimeUtc'] = Carbon::now()->subDay()->toDateTimeString();
        $jsonObject['SellingStatus']['BidCount'] = 0;
        $jsonObject['SellingStatus']['ListingStatus'] = 'Completed';
        $ebayProductUnsold->json_object = $jsonObject;
        $ebayProductUnsold->save();
        $ebayProductUnsold->refresh();

        // Ended early
        ProductListing::factory()->create([
            'product_id' => $product->id,
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'document_id' => $ebayProductEndedEarly->id,
            'document_type' => EbayLegacyProduct::class,
        ]);

        // Sold
        ProductListing::factory()->create([
            'product_id' => $product->id,
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'document_id' => $ebayProductSold->id,
            'document_type' => EbayLegacyProduct::class,
        ]);

        // Unsold - should be relisted
        ProductListing::factory()->create([
            'product_id' => $product->id,
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'document_id' => $ebayProductUnsold->id,
            'document_type' => EbayLegacyProduct::class,
        ]);

        EbayProductSetting::factory()->create([
            'product_id' => $product->id,
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
            'auction_until_sold_enabled' => true,
            'auction_until_sold_price_reduction_percentage' => 10,
            'auction_until_sold_minimum_price' => 50.00,
        ]);

        $this->manager->relistAuctions();

        $relistedEbayProduct = EbayLegacyProduct::where('ItemID', '355793538336')->first();

        $this->assertNotNull($relistedEbayProduct, 'The product was not relisted');

        $relistedProductListing = ProductListing::where('document_id', $relistedEbayProduct->id)->first();

        $this->assertDatabaseHas((new EbayLegacyProduct())->getTable(), [
            'ItemID' => '355793538336',
            'SKU' => 'testMBV3.0',
        ]);

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'product_id' => $product->id,
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'document_id' => $relistedEbayProduct->id,
            'document_type' => EbayLegacyProduct::class,
        ]);

        $this->assertEquals('123', $relistedProductListing->metadata['originalItemId']);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_it_can_give_friendly_error_if_ebay_legacy_product_doesnt_have_a_sku(): void
    {
        EbayLegacyProduct::factory()->create([
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
        ]);

        DataImportMapping::factory()->create([
            'model' => EbayLegacyProduct::class,
            'integration_instance_id' => $this->ebayIntegrationInstance->id,
            'mapping' => [
                [
                    'listing_field' => 'CurrentPriceValue',
                    'sku_field' => 'price.Retail.value',
                    'parsers' => []
                ],
                [
                    'listing_field' => 'SKU',
                    'sku_field' => 'sku',
                    'parsers' => []
                ],
            ]
        ]);

        $manager = new EbayLegacyProductManager($this->ebayIntegrationInstance);

        $this->assertThrows(
            fn () => $manager->createSkuProducts(
                createForAll: true,
            ),
            SalesChannelProductMissingSkuException::class,
        );
    }
}
