<?php

namespace Modules\Ebay\Tests\Feature\Controllers;

use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\User;
use App\Models\Warehouse;
use Laravel\Sanctum\Sanctum;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Entities\EbayOrder;
use Modules\Ebay\Tests\EbayMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class EbayFulfillmentControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use EbayMockRequests;

    private EbayIntegrationInstance $ebayIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();
        $this->ebayIntegrationInstance = EbayIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create();

        $this->mockGetAccessToken();

        Sanctum::actingAs(User::first());
    }

    public function test_ebay_fulfillment_controller(): void
    {
        $this->mockFulfillOrder();

        $warehouse = Warehouse::first();

        $ebayOrder = EbayOrder::factory([
            'integration_instance_id' => $this->ebayIntegrationInstance->id
            ])
            ->hasOrderItems()
            ->create();

        $salesOrder = SalesOrder::factory([
            'sales_channel_id' => $this->ebayIntegrationInstance->salesChannel->id,
            'sales_order_number' => $ebayOrder->orderId,
            'sales_channel_order_id' => $ebayOrder->id,
            'sales_channel_order_type' => EbayOrder::class
        ])
            ->has(SalesOrderLine::factory()->state([
                'sales_channel_line_id' => $ebayOrder->orderItems->first()->lineItemId,
                'warehouse_id' => $warehouse->id,
            ]))
            ->create();

        $fulfillmentDate = now();

        $salesOrderFulfillment = SalesOrderFulfillment::factory([
            'sales_order_id' => $salesOrder->id,
            'warehouse_id' => $warehouse->id,
            'fulfilled_shipping_method' => 'USPS',
            'fulfilled_at' => $fulfillmentDate,
            'tracking_number' => '123456789',
        ])->has(SalesOrderFulfillmentLine::factory(1)->state([
            'sales_order_line_id' => $salesOrder->salesOrderLines->first()->id,
        ]))
        ->create();

        $salesOrderFulfillment->refresh();

        $this->postJson(route('ebay.fulfill', [
            'integrationInstance' => $this->ebayIntegrationInstance->id,
        ]), [
            'sales_order_fulfillment_id' => $salesOrderFulfillment->id,
        ])->assertStatus(200);

        $this->assertNotNull($salesOrderFulfillment->refresh()->submitted_to_sales_channel_at);
    }
}