<?php

declare(strict_types=1);

namespace Modules\Ebay\Services;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Data\UpdateProductListingSalesChannelQtyData;
use App\Data\UpdateProductListingSalesChannelQtyItemData;
use App\Enums\HttpMethodEnum;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\View;
use Modules\Ebay\ApiDataTransferObjects\EbayGetLegacyProductsAdt;
use Modules\Ebay\ApiDataTransferObjects\EbayLegacyGetOrdersAdt;
use Modules\Ebay\ApiDataTransferObjects\EbayReviseItemAdt;
use Modules\Ebay\ApiDataTransferObjects\EbaySyncInventoryAdt;
use Modules\Ebay\Data\EbayLegacyProductData;
use Modules\Ebay\Data\EbayResponseData;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\Ebay\Enums\EbayLegacyGetProductActionsEnum;
use Modules\Ebay\Exceptions\EbayBadRequestException;
use Modules\Ebay\Services\Transformer\XmlResponseTransformer;
use Spatie\LaravelData\DataCollection;

/**
 * Class TradingClient
 */
class EbayTradingClient extends EbayAuthenticationClient
{
    const MAX_RESULTS = 100;

    public static function withAuthClient(EbayAuthenticationClient $authClient)
    {
        return new static($authClient);
    }

    /**
     * @throws Exception
     */
    public function getProducts(EbayGetLegacyProductsAdt|ApiDataTransformerInterface $parameters): EbayResponseData
    {
        // If no product records found, we should get all products
        if (EbayLegacyProduct::count() == 0)
        {
            $parameters->all = true;
        }

        $requestParams = $parameters->transform();

        // get view template
        $body = View::make(
            "ebay::trading.{$parameters->action->value}",
            $requestParams
        )->render();

        $response = $this->requestLegacy(HttpMethodEnum::POST, $parameters->action->value, $body);

        $responseXml = $response->body();
        $simpleXml = simplexml_load_string($responseXml);
        $xmlAsArray = XmlResponseTransformer::simplexmlToArray($simpleXml);

        if ($xmlAsArray['Ack'] != 'Success') {
            throw new EbayBadRequestException('Ebay API error: '.$xmlAsArray['Errors']['LongMessage']);
        }

        if ($parameters->action == EbayLegacyGetProductActionsEnum::GET_MY_EBAY_SELLING) {
            $data = $xmlAsArray['ActiveList']['ItemArray']['Item'];
            if (!isset($data[0])) {
                $data = [$data];
            }
            $totalPages = $xmlAsArray['ActiveList']['PaginationResult']['TotalNumberOfPages'];
        } elseif ($parameters->action == EbayLegacyGetProductActionsEnum::GET_SELLER_LIST) {
            if(empty($xmlAsArray['ItemArray']))
            {
                return EbayResponseData::from([
                    'collection' => EbayLegacyProductData::collection([]),
                    'totalPages' => 0,
                    'currentPage' => 1,
                    'nextPage' => null,
                ]);
            }
            $data = $xmlAsArray['ItemArray']['Item'];
            if (!isset($data[0])) {
                $data = [$data];
            }
            $totalPages = $xmlAsArray['PaginationResult']['TotalNumberOfPages'];
        } else {
            throw new Exception('Unknown ebay get products action: '.$parameters->action->value);
        }

        return EbayResponseData::from([
            'collection' => EbayLegacyProductData::collection($data),
            'totalPages' => $totalPages,
            'currentPage' => $requestParams['pageNumber'],
            'nextPage' => ($requestParams['pageNumber'] < $totalPages) ? $requestParams['pageNumber'] + 1 : null,
        ]);
    }

    public function getInventoryItems($page = 1)
    {
        $callName = 'GetSellerList';
        $url = config('ebay.trading.domain');
        $request = $this->authClient->createLegacyRequest($callName);

        $startTimeTo = Carbon::now()->subSecond();
        $startTimeFrom = $startTimeTo->subDays(120);

        // get view template
        $body = View::make(
            "ebay::trading.{$callName}",
            [
                'limit' => self::MAX_RESULTS,
                'pageNumber' => $page,
                'startTimeFrom' => $startTimeFrom->toIso8601ZuluString(),
                'startTimeTo' => $startTimeTo->toIso8601ZuluString(),
            ]
        )->render();
        $request->withBody($body, 'text/xml');

        $response = $request->post($url);
        $responseXml = $response->body();
        dd($responseXml);

        $simpleXml = simplexml_load_string($responseXml);
        $xmlAsArray = XmlResponseTransformer::simplexmlToArray($simpleXml);

        //        $skus = [];
        //        foreach ($xmlAsArray['ActiveList']['ItemArray']['Item'] as $item) {
        //            $skus[] = $item['SKU'] ?? null;
        //        }

        $items = array_map(function ($item) {
            $entity = [
                'title' => $item['Title'],
                'ebay_item_id' => $item['ItemID'],
            ];

            if (isset($item['SKU'])) {
                $entity['sku'] = $item['SKU'];
            }

            return [
                'entity' => $entity,
                'json_object' => $item,
            ];
        }, $xmlAsArray['ActiveList']['ItemArray']['Item']);

        $ebayProductCollection = new Collection();

        if (count($items) === 0) {
            return EbayResponseData::from([
                'collection' => $ebayProductCollection,
                'nextToken' => 0,
                'totalPages' => 0,
                'currentPage' => $page,
            ]);
        }

        $totalRecords = $xmlAsArray['ActiveList']['PaginationResult']['TotalNumberOfEntries'];
        $totalPages = $xmlAsArray['ActiveList']['PaginationResult']['TotalNumberOfPages'];
        $nextPage = $page + 1;

        foreach ($items as $item) {
            $product = EbayLegacyProductData::from([
                ...$item['entity'],
                ...['json_object' => $item['json_object']],
            ]);
            $ebayProductCollection->add(
                item: $product,
            );
        }

        return EbayResponseData::from([
            'collection' => $ebayProductCollection,
            'nextToken' => $nextPage,
            'totalPages' => (int) $totalPages,
            'currentPage' => (int) $page,
        ]);
    }

    /**
     * @throws Exception
     */
    public function getOrders(EbayLegacyGetOrdersAdt|ApiDataTransformerInterface $parameters): EbayResponseData
    {
        $parameters = $parameters->transform();
        $parameters['limit'] = self::MAX_RESULTS;
        $parameters['page'] = 1;

        $callName = 'GetSellerTransactions';
        // get view template
        $body = View::make(
            "ebay::trading.{$callName}",
            $parameters
        )->render();

        $response = $this->requestLegacy(HttpMethodEnum::POST, $callName, $body);

        $responseXml = $response->body();

        $simpleXml = simplexml_load_string($responseXml);
        $xmlAsArray = XmlResponseTransformer::simplexmlToArray($simpleXml);

        // TODO: Test new api vs old api to compare and decide what to use for orders
        // TODO: Probably need to drop the ebay_orders table and create a new structure (definitely if using the old api)
        dd($xmlAsArray);
        $collection = new Collection();

        $items = array_map(function ($item) {
            // Order
            $order = [
                'orderId' => $item['ContainingOrder']['OrderID'],
                'creationDate' => $item['CreatedDate'],
                'lastModifiedDate' => $item['Status']['LastTimeModified'],
                'orderPaymentStatus' => $item['Status']['CompleteStatus'],
                'orderFulfillmentStatus' => $item['Status']['CompleteStatus'],
                'json_object' => $item,
            ];

            $orderLineItem = [
                'ebay_item_id' => $item['Item']['ItemID'],
                'quantity' => $item['Item']['Quantity'],
            ];

            if (isset($item['Item']['SKU'])) {
                $orderLineItem['sku'] = $item['Item']['SKU'];
            }

            return [
                'entity' => $order,
                'json_object' => $item,
            ];
        }, $xmlAsArray['TransactionArray']['Transaction']);

        if (count($items) === 0) {
            return EbayResponseData::from([
                'collection' => $collection,
                'nextToken' => 0,
                'totalPages' => 0,
                'currentPage' => $page,
            ]);
        }

        $totalRecords = $xmlAsArray['ActiveList']['PaginationResult']['TotalNumberOfEntries'];
        $totalPages = $xmlAsArray['ActiveList']['PaginationResult']['TotalNumberOfPages'];
        $nextPage = $page + 1;

        foreach ($items as $item) {
            $product = EbayLegacyProductData::from([
                ...$item['entity'],
                ...['json_object' => $item['json_object']],
            ]);
            $ebayProductCollection->add(
                item: $product,
            );
        }

        return EbayResponseData::from([
            'collection' => $ebayProductCollection,
            'nextToken' => $nextPage,
            'totalPages' => (int) $totalPages,
            'currentPage' => (int) $page,
        ]);
    }

    /**
     * @throws Exception
     */
    public function syncInventory(EbaySyncInventoryAdt|ApiDataTransformerInterface $parameters): DataCollection
    {
        $payloads = $parameters->transform();

        $successfulProductListingUpdates = collect();

        foreach ($payloads as $payload) {
            // get view template
            $body = View::make(
                "ebay::trading.ReviseInventoryStatus",
                [
                    'payload' => $payload,
                ]
            )->render();
            $response = $this->requestLegacy(HttpMethodEnum::POST, 'ReviseInventoryStatus', $body);

            $responseXml = $response->body();

            $simpleXml = simplexml_load_string($responseXml);
            $xmlAsArray = XmlResponseTransformer::simplexmlToArray($simpleXml);

            if ($xmlAsArray['Ack'] != 'Success') {
                throw new EbayBadRequestException('Ebay API error: '.$xmlAsArray['Errors']['LongMessage']);
            }

            foreach ($payload as $item) {
                $successfulProductListingUpdates->add(UpdateProductListingSalesChannelQtyData::from([
                    'sales_channel_qty_last_updated' => now(),
                    'id' => $item['id'],
                    'sales_channel_qty' => $item['Quantity'],
                ]));
            }
        }
        return UpdateProductListingSalesChannelQtyData::collection($successfulProductListingUpdates);
    }

    /**
     * @throws Exception
     */
    public function publish(string $payload, ?string $siteId): array
    {
        if (!is_null($siteId)) {
            $this->setSiteId($siteId);
        }
        $response = $this->requestLegacy(HttpMethodEnum::POST, 'AddItem', $payload);

        $responseXml = $response->body();

        $simpleXml = simplexml_load_string($responseXml);
        $xmlAsArray = XmlResponseTransformer::simplexmlToArray($simpleXml);

        $ack = $xmlAsArray['Ack'];
        if ($ack == 'Failure') {
            $errors = Arr::wrap($xmlAsArray['Errors']);

            // Handle multiple errors if they exist
            $message = $errors['LongMessage'] ?? implode(', ', array_column($errors, 'LongMessage'));
            throw new EbayBadRequestException('Ebay API error(s): '.$message);
        }

        return $xmlAsArray;
    }

    /**
     * @throws EbayBadRequestException
     */
    public function uploadSiteHostedPictures(string $url): array
    {
        $payload = View::make(
            "ebay::trading.UploadSiteHostedPictures",
            [
                'ExternalPictureURL' => $url,
            ]
        )->render();

        $response = $this->requestLegacy(HttpMethodEnum::POST, 'UploadSiteHostedPictures', $payload);

        $responseXml = $response->body();

        $simpleXml = @simplexml_load_string($responseXml);
        if ($simpleXml === false) {
            throw new EbayBadRequestException('Ebay API error UploadSiteHostedPictures: '.$responseXml);
        }

        $xmlAsArray = XmlResponseTransformer::simplexmlToArray($simpleXml);

        $ack = $xmlAsArray['Ack'];
        if ($ack == 'Failure') {
            $errors = Arr::wrap($xmlAsArray['Errors']);

            // Handle multiple errors if they exist
            $message = $errors['LongMessage'] ?? implode(', ', array_column($errors, 'LongMessage'));
            throw new EbayBadRequestException('Ebay API error(s) UploadSiteHostedPictures for ' . $url . ': '.$message);
        }

        return $xmlAsArray;
    }

    /**
     * @throws EbayBadRequestException
     * @throws Exception
     */
    public function getItem(string $ItemID): array
    {
        $payload = View::make(
            "ebay::trading.GetItem",
            [
                'ItemID' => $ItemID,
            ]
        )->render();

        $response = $this->requestLegacy(HttpMethodEnum::POST, 'GetItem', $payload);

        $responseXml = $response->body();

        $simpleXml = simplexml_load_string($responseXml);
        $xmlAsArray = XmlResponseTransformer::simplexmlToArray($simpleXml);

        if (!isset($xmlAsArray['Ack'])) {
            throw new EbayBadRequestException('Fatal Ebay API error: '.json_encode($xmlAsArray));
        }

        if ($xmlAsArray['Ack'] != 'Success') {
            throw new EbayBadRequestException('Ebay API error: '.$xmlAsArray['Errors']['LongMessage']);
        }

        return $xmlAsArray;
    }

    /**
     * @throws Exception
     */
    public function getSuggestedCategories(string $query): array
    {
        $body = View::make(
            "ebay::trading.GetSuggestedCategories",
            [
                'query' => $query,
            ]
        )->render();

        $response = $this->requestLegacy(HttpMethodEnum::POST, 'GetSuggestedCategories', $body);

        $responseXml = $response->body();

        $simpleXml = simplexml_load_string($responseXml);
        $xmlAsArray = XmlResponseTransformer::simplexmlToArray($simpleXml);

        if ($xmlAsArray['Ack'] != 'Success') {
            throw new EbayBadRequestException('Ebay API error: '.$xmlAsArray['Errors']['LongMessage']);
        }

        return $xmlAsArray;
    }

    /**
     * @throws EbayBadRequestException
     * @throws Exception
     */
    public function relistItem(EbayReviseItemAdt $parameters): array
    {
        $this->setSiteId($parameters->Site);

        $body = View::make(
            "ebay::trading.RelistItem",
            [
                'ItemID' => $parameters->ItemID,
                'StartPrice' => $parameters->StartPrice,
            ]
        )->render();

        $response = $this->requestLegacy(HttpMethodEnum::POST, 'RelistItem', $body);

        $responseXml = $response->body();

        $simpleXml = simplexml_load_string($responseXml);
        $xmlAsArray = XmlResponseTransformer::simplexmlToArray($simpleXml);

        $ack = $xmlAsArray['Ack'];
        if ($ack == 'Failure') {
            $errors = Arr::wrap($xmlAsArray['Errors']);

            // Handle multiple errors if they exist
            $message = $errors['LongMessage'] ?? implode(', ', array_column($errors, 'LongMessage'));
            throw new EbayBadRequestException('Ebay API error(s): '.$message);
        }

        return $xmlAsArray;
    }
}
