<?php

declare(strict_types=1);

namespace Modules\Ebay\Services;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\ClientResponseDataInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\SalesChannelClientInterface;
use Exception;
use Illuminate\Support\Collection;
use Modules\Ebay\ApiDataTransferObjects\EbayReviseItemAdt;
use Modules\Ebay\Exceptions\EbayBadRequestException;
use Modules\Ebay\Services\Api\Inventory\InventoryClient;
use Spatie\LaravelData\DataCollection;

/**
 * A composition of various eBay APIs that can be called.
 * The "Client" functions as a facade. Adding functionality is as easy as adding a function
 * to this class to call the underlying API client.
 */
class EbayClient extends EbayAuthenticationClient implements SalesChannelClientInterface
{
    //    public function __construct(
    //        protected EbayAuthenticationClient $authClient,
    //        protected InventoryClient          $inventoryClient,
    //        protected FulfillmentClient        $fulfillmentClient,
    //        protected TradingClient            $tradingClient,
    //        protected bool                     $useLegacyApis = true
    //    ) {}

    private InventoryClient $inventoryClient;

    private EbayFulfillmentClient $fulfillmentClient;

    private EbayTradingClient $tradingClient;

    private bool $useLegacyApis = true;

    public function __construct(IntegrationInstanceInterface $integrationInstance)
    {
        parent::__construct($integrationInstance);

        //$this->inventoryClient = new InventoryClient($integrationInstance);
        $this->fulfillmentClient = new EbayFulfillmentClient($integrationInstance);
        $this->tradingClient = new EbayTradingClient($integrationInstance);
    }

    //
    //    public function getInventoryItemsNeedingMigration()
    //    {
    //        return $this->tradingClient->getInventoryItemsNeedingMigration();
    //    }

    //    public function getOrders(\DateTimeInterface $startDate, \DateTimeInterface $endDate): EbayResponseDto
    //    {
    //        if ($this->useLegacyApis) {
    //            return $this->tradingClient->getOrders($startDate, $endDate);
    //        } else {
    //            return $this->fulfillmentClient->getOrders($startDate, $endDate);
    //        }
    //    }

    //    public function getInventoryItems($page = 1): EbayResponseDto
    //    {
    //        if ($this->useLegacyApis) {
    //            return $this->tradingClient->getInventoryItems($page);
    //        } else {
    //            return $this->inventoryClient->getInventoryItems($page);
    //        }
    //    }

    /**
     * @throws Exception
     */
    public function getOrders(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        //$this->tradingClient->getOrders($parameters);
        return $this->fulfillmentClient->getOrders($parameters);
    }

    public function getOrder(string $orderId): ClientResponseDataInterface
    {
        return $this->fulfillmentClient->getOrder($orderId);
    }

    /**
     * @throws EbayBadRequestException
     */
    public function getProduct(string $productId): array
    {
        return $this->tradingClient->getItem($productId);
    }

    /**
     * @throws Exception
     */
    public function getProducts(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->tradingClient->getProducts($parameters);
    }

    /**
     * @throws Exception
     */
    public function getLegacyProducts(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->tradingClient->getProducts($parameters);
    }

    /**
     * @throws Exception
     */
    public function fulfillOrder(ApiDataTransformerInterface $parameters): bool
    {
        return $this->fulfillmentClient->fulfillOrder($parameters);
    }

    public function fulfillOrders(ApiDataTransformerInterface $parameters)
    {
        // TODO: Implement fulfillOrders() method.
    }

    public function syncInventory(ApiDataTransformerInterface $parameters): DataCollection
    {
        return $this->tradingClient->syncInventory($parameters);
    }

    /**
     * @throws Exception
     */
    public function publish(string $payload, ?string $siteId): array
    {
        return $this->tradingClient->publish($payload, $siteId);
    }

    public function getSuggestedCategories(string $query): array
    {
        return $this->tradingClient->getSuggestedCategories($query);
    }

    public function uploadSiteHostedPictures($url): array
    {
        return $this->tradingClient->uploadSiteHostedPictures($url);
    }

    public function relistItem(EbayReviseItemAdt $parameters): array
    {
        return $this->tradingClient->relistItem($parameters);
    }
}
