<?php

declare(strict_types=1);

namespace Modules\Ebay\Services;

use App\Abstractions\Integrations\AbstractIntegrationClient;
use App\Abstractions\Integrations\Concerns\OauthAuthenticationTrait;
use App\Abstractions\Integrations\Data\OauthRefreshTokenResponseData;
use App\Abstractions\Integrations\IntegrationClientInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\OauthAuthenticationClientInterface;
use App\Enums\HttpMethodEnum;
use App\Exceptions\OauthRefreshTokenFailureException;
use App\Models\ApiLog;
use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7\MultipartStream;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Http\Client\PendingRequest;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use Modules\Ebay\ValueObject\Oauth\State;
use Ramsey\Uuid\Uuid;
use Storage;

/**
 * Class AuthClient
 */
class EbayAuthenticationClient extends AbstractIntegrationClient implements IntegrationClientInterface, OauthAuthenticationClientInterface
{
    use OauthAuthenticationTrait;

    const EBAY_TOKEN_ARRAY_SHAPE = [
        'access_token' => 'string',
        'expires_in' => 'int',
        'refresh_token' => 'string',
        'refresh_token_expires_in' => 'int',
        'token_type' => 'string',
    ];

    const CONTENT_TYPE = 'application/json';

    private string $clientId;

    private string $devId;

    private string $clientSecret;

    private string $path;

    private string $httpMethod;

    private string $siteId = '0';

    // Access token generated from refresh token.
    private ?string $accessToken = null;

    // Constructor is DI so we can unit test this
    public function __construct(IntegrationInstanceInterface $integrationInstance)
    {
        parent::__construct($integrationInstance);
        $this->clientId = config('ebay.auth.app_id');
        $this->devId = config('ebay.auth.dev_id');
        $this->clientSecret = config('ebay.auth.cert_id');

        if ($integrationInstance->refresh_token) {
            $this->setClientCredentials($integrationInstance->refresh_token);
        }
    }

    public function client(): PendingRequest
    {
        // Not implemented for eBay
    }

    public function getRedirectUrl(): string
    {
        $scopes = implode(' ', config('ebay.auth.scopes'));

        $params = [
            'state' => config('app.url') . '_' . $this->integrationInstance->id,
            'redirect_uri' => config('ebay.auth.redirect_uri'),
            'scope' => $scopes,
            'client_id' => config('ebay.auth.app_id'),
            'response_type' => 'code',
        ];

        return config('ebay.auth.user_authorization_url').'?'.urldecode(http_build_query($params));
    }

    public function getRefreshToken(): string
    {
        return $this->refreshToken;
    }

    /**
     * @throws OauthRefreshTokenFailureException
     */
    public function getRefreshTokenFromAuthCode(string $authCode): OauthRefreshTokenResponseData
    {
        $headers = [
            'Authorization' => 'Basic '.
                base64_encode(config('ebay.auth.app_id').':'.config('ebay.auth.cert_id')),
        ];

        $response = Http::asForm()
            ->withHeaders($headers)
            ->post(
                config('ebay.tokens.token_url'),
                [
                    'grant_type' => 'authorization_code',
                    'code' => $authCode,
                    'redirect_uri' => config('ebay.auth.redirect_uri'),
                ]
            );

        if (!$response->ok()) {
            throw new OauthRefreshTokenFailureException($response->body());
        }

        return OauthRefreshTokenResponseData::from($response->json());
    }

    /**
     * @throws Exception
     */
    public function getAccessToken(): string
    {
        if (is_null($this->accessToken)) {
            $headers = [
                'Authorization' => 'Basic '.
                    base64_encode(config('ebay.auth.app_id').':'.config('ebay.auth.cert_id')),
            ];

            $requestBody = [
                'grant_type' => 'refresh_token',
                'refresh_token' => $this->getRefreshToken(),
            ];

            if ($this->isApiLoggingEnabled) {
                $apiLog = ApiLog::create([
                    'integration_instance_id' => $this->integrationInstance->id,
                    'url' => config('ebay.tokens.token_url'),
                    'requestHeaders' => $headers,
                    'requestBody' => $requestBody,
                    'responseStatusCode' => null,
                    'responseHeaders' => null,
                    'responseBody' => null,
                ]);
            }

            $response = Http::asForm()
                ->withHeaders($headers)
                ->post(
                    config('ebay.tokens.token_url'),
                    $requestBody,
                );

            if ($this->isApiLoggingEnabled) {
                $apiLog->update([
                    'responseStatusCode' => $response->status(),
                    'responseHeaders' => $response->headers(),
                    'responseBody' => $response->body(),
                ]);
            }

            if ($response->ok()) {
                $this->accessToken = $response->json()['access_token'];
            } else {
                throw new Exception('Invalid ebay authorization code or state.');
            }
        }

        // TODO: Are we generating a fresh token every time?  Should we be caching this and using the expires in (see git history for this class)
        return $this->accessToken;
    }

    /**
     * @throws Exception
     */
    public function createRequest(HttpMethodEnum $httpMethod, string $path, array|string $parameters): Response
    {
        $this->path = config('ebay.api.domain').$path;
        $headers = [
            'Authorization' => 'Bearer '.$this->getAccessToken(),
            'Content-Type' => self::CONTENT_TYPE,
            'Accept' => self::CONTENT_TYPE,
        ];

        $apiLog = null;

        if ($this->isApiLoggingEnabled) {
            $apiLog = ApiLog::create([
                'integration_instance_id' => $this->integrationInstance->id,
                'url' => $this->path.'?'.http_build_query($parameters),
                'requestHeaders' => $headers,
                'requestBody' => '',
                'responseStatusCode' => null,
                'responseHeaders' => null,
                'responseBody' => null,
            ]);
        }

        $response = Http::withHeaders($headers)
            ->{$httpMethod->value}($this->path.'?'.http_build_query($parameters));

        if ($this->isApiLoggingEnabled) {
            $apiLog->update([
                'responseStatusCode' => $response->status(),
                'responseHeaders' => $response->headers(),
                'responseBody' => $response->body(),
            ]);
        }

        return $response;
    }

    /**
     * @throws Exception
     */
    public function createLegacyRequest(HttpMethodEnum $httpMethod, string $callName, string $body): Response
    {
        $this->path = config('ebay.trading.domain');

        $headers = [
            'X-EBAY-API-IAF-TOKEN' => $this->getAccessToken(),
            'X-EBAY-API-CALL-NAME' => $callName,
            'X-EBAY-API-SITEID' => $this->siteId,
            'X-EBAY-API-COMPATIBILITY-LEVEL' => config('ebay.trading.compatibility_level'),
        ];

        $apiLog = null;

        if ($this->isApiLoggingEnabled) {
            $apiLog = ApiLog::create([
                'integration_instance_id' => $this->integrationInstance->id,
                'url' => $this->path.' '.$callName,
                'requestHeaders' => $headers,
                'requestBody' => $body,
                'responseStatusCode' => null,
                'responseHeaders' => null,
                'responseBody' => null,
            ]);
        }

        if ($callName == 'UploadSiteHostedPictures')
        {
            $xmlPayload = simplexml_load_string($body);
            $imagePath = (string) $xmlPayload->ExternalPictureURL;
            $request = Http::withHeaders($headers);
            $path = $this->path;

            if (!str_starts_with($imagePath, 'http')) {
                $filename = basename($imagePath);
                unset($xmlPayload->ExternalPictureURL);
                $imageContents = Storage::disk('images')->get($imagePath);

                if($imageContents) {
                    $request->attach(
                        name: 'file',
                        contents: $imageContents,
                        filename: $filename
                    );
                }

                $response = $request->{$httpMethod->value}($path, [
                    'form_params' => [
                        'name' => 'XML Payload',
                        'contents' => $xmlPayload->asXML(),
                        'headers' => ['Content-Disposition' => 'form-data; name="XML Payload"']
                    ]
                ]);
            } else {
                $response = $request->withBody($body, 'text/xml')->{$httpMethod->value}($path);
            }
        } else {
            $headers['Content-Type'] = self::CONTENT_TYPE;
            $response = Http::withHeaders($headers)->withBody($body, 'text/xml')->{$httpMethod->value}($this->path);
        }

        if ($this->isApiLoggingEnabled) {
            $apiLog->update([
                'responseStatusCode' => $response->status(),
                'responseHeaders' => $response->headers(),
                'responseBody' => $response->body(),
            ]);
        }

        return $response;
    }

    public function setSiteId(string $siteId): void
    {
        $this->siteId = $siteId;
    }

    /**
     * @throws Exception
     */
    public function requestLegacy(HttpMethodEnum $httpMethod, string $action, string $body): Response
    {
        return $this->createLegacyRequest($httpMethod, $action, $body);
    }

    /**
     * @throws Exception
     */
    public function request(HttpMethodEnum $httpMethod, string $path, array $parameters): Response
    {
        return $this->createRequest($httpMethod, $path, $parameters);
    }

    public function pushRequest(HttpMethodEnum $httpMethod, string $path, array $body): Response
    {
        $this->path = config('ebay.api.domain').$path;
        $headers = [
            'Authorization' => 'Bearer '.$this->getAccessToken(),
            'Content-Type' => self::CONTENT_TYPE,
            'Accept' => self::CONTENT_TYPE,
        ];

        $apiLog = null;

        if ($this->isApiLoggingEnabled) {
            $apiLog = ApiLog::create([
                'integration_instance_id' => $this->integrationInstance->id,
                'url' => $this->path,
                'requestHeaders' => $headers,
                'requestBody' => json_encode($body),
                'responseStatusCode' => null,
                'responseHeaders' => null,
                'responseBody' => null,
            ]);
        }

        $response = Http::withHeaders($headers)
            ->{$httpMethod->value}($this->path, $body);

        if ($this->isApiLoggingEnabled) {
            $apiLog->update([
                'responseStatusCode' => $response->status(),
                'responseHeaders' => $response->headers(),
                'responseBody' => $response->body(),
            ]);
        }

        return $response;
    }
}
