<?php

declare(strict_types=1);

namespace Modules\Ebay\Services\Api\Inventory;

use Modules\Ebay\Data\EbayProductData;
use Modules\Ebay\Data\EbayResponseData;
use Modules\Ebay\Services\EbayAuthenticationClient;

/**
 * Class LocationApi
 */
class InventoryClient
{
    const MAX_RESULTS = 100;

    public function __construct(
        protected EbayAuthenticationClient $authClient,
        protected InventoryMock $inventoryMock
    ) {
    }

    public static function withAuthClient(EbayAuthenticationClient $authClient)
    {
        return new static(
            $authClient,
            new InventoryMock()
        );
    }

    /**
     * Returns a string as a heredoc for the response.
     */
    public function getInventoryItems(
        $limit = self::MAX_RESULTS,
        $offset = 0
    ): EbayResponseData {
        if ($limit > self::MAX_RESULTS) {
            throw new \InvalidArgumentException(
                'Limit cannot be greater than '.self::MAX_RESULTS
            );
        }

        $path = config('ebay.api.inventory.endpoints.inventory');
        $request = $this->authClient->createRequest();
        $queryParams = [
            'limit' => $limit,
            'offset' => $offset,
        ];

        $response = $request->get($path, $queryParams);

        if ($response['total'] === 0) {
            return EbayResponseData::from([
                'collection' => EbayProductData::collection([]),
                'nextToken' => (string) $response['total'],
            ]);
        }

        return EbayResponseData::from([
            'collection' => EbayProductData::collection($response['inventoryItems']),
            'nextToken' => (string) $response['total'],
        ]);
    }
}
