<?php

declare(strict_types=1);

namespace Modules\Ebay\Repositories;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProductRepository;
use Carbon\Carbon;
use Illuminate\Support\Collection;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\Ebay\Enums\EbayListingTypeCodeTypeEnum;

/**
 * EbayLegacyProductRepository
 *
 */
class EbayLegacyProductRepository extends AbstractSalesChannelProductRepository
{
    public function getModelClassName(): string
    {
        return EbayLegacyProduct::class;
    }

    public function getEbayProductsEligibleForRelist(IntegrationInstanceInterface $integrationInstance): Collection
    {
        $listings = EbayLegacyProduct::with('productListing.product.ebayProductSettings')
            ->whereHas('productListing', function ($query) use ($integrationInstance) {
                $query->whereHas('product', function ($query) use ($integrationInstance) {
                    $query->whereHas('ebayProductSettings', function ($query) use ($integrationInstance) {
                        $query->where('auction_until_sold_enabled', true);
                        $query->where('integration_instance_id', $integrationInstance->id);
                    });
                    // In stock
                    $query->whereHas('productInventory', function ($query) {
                        $query->where('warehouse_id', 0);
                        $query->where('inventory_available', '>', 0);
                    });
                });
            })
            ->where('ListingType', EbayListingTypeCodeTypeEnum::CHINESE)
            ->where('ListingStatus', 'Completed')
            ->where('BidCount', 0) // Unsold
            ->whereNull('EndingReason')
            ->get();

        $activeEbayChineseListings = $this->getActiveEbayChineseListings($integrationInstance);

        // Reject if listing product ID is in active ebay chinese listings
        $listings = $listings->reject(function ($listing) use ($activeEbayChineseListings) {
            return $activeEbayChineseListings->contains('productListing.product_id', $listing->productListing->product_id);
        });

        // Group by product ID and select the most recent listing for each product
        return $listings->groupBy('productListing.product_id')->map(function ($group) {
            return $group->sortByDesc('EndTimeUtc')->first();
        });
    }

    public function getActiveEbayChineseListings($integrationInstance): Collection
    {
        return EbayLegacyProduct::with('productListing.product')
            ->where('integration_instance_id', $integrationInstance->id)
            ->where('ListingType', EbayListingTypeCodeTypeEnum::CHINESE)
            ->where('ListingStatus', 'Active')
            ->get();
    }

    public function getStartDateForNew(IntegrationInstanceInterface $integrationInstance): ?Carbon
    {
        if (!EbayLegacyProduct::getLastModified()) {
            return null;
        }

        $startDate = EbayLegacyProduct::where('integration_instance_id', $integrationInstance->id)
            ->latest(EbayLegacyProduct::getLastModified())
            ->pluck(EbayLegacyProduct::getLastModified())
            ->first();

        $startDateForActiveAuction = EbayLegacyProduct::where('integration_instance_id', $integrationInstance->id)
            ->where('ListingType', EbayListingTypeCodeTypeEnum::CHINESE)
            ->where('ListingStatus', 'Active')
            ->oldest('StartTimeUtc')
            ->pluck('StartTimeUtc')
            ->first();

        if ($startDateForActiveAuction) {
            return Carbon::parse($startDateForActiveAuction);
        }

        return $startDate ? Carbon::parse($startDate)->addSecond() : null;
    }
}
