<?php

declare(strict_types=1);

namespace Modules\Ebay\Managers;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\ClientResponseDataInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProductManager;
use App\Abstractions\Integrations\SalesChannels\SalesChannelProductRepositoryInterface;
use App\Data\SalesChannelHydratedFieldData;
use App\DTO\ProductListingDto;
use App\Enums\SalesChannelProductTemplateFieldTypeEnum;
use App\Managers\ProductManager;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannelProductTemplate;
use App\Models\SalesChannelProductTemplateField;
use App\Repositories\IntegrationInstanceRepository;
use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\View;
use Modules\Ebay\ApiDataTransferObjects\EbayReviseItemAdt;
use Modules\Ebay\ApiDataTransferObjects\EbaySyncInventoryAdt;
use Modules\Ebay\Data\EbayLegacyProductData;
use Modules\Ebay\Data\PublishProductListingResponseData;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Entities\EbayLegacyProduct;
use Modules\Ebay\Repositories\EbayLegacyProductRepository;
use Modules\Ebay\Services\EbayClient;

class EbayLegacyProductManager extends AbstractSalesChannelProductManager
{
    private EbayLegacyProductRepository $ebayProductsRepository;
    private IntegrationInstanceRepository $integrationInstances;


    public function __construct(EbayIntegrationInstance|IntegrationInstanceInterface $ebayIntegrationInstance)
    {
        $this->ebayProductsRepository = app(EbayLegacyProductRepository::class);
        $this->integrationInstances = app(IntegrationInstanceRepository::class);
        $this->setModel(new EbayLegacyProduct());

        parent::__construct($ebayIntegrationInstance, new EbayClient($ebayIntegrationInstance), $this->ebayProductsRepository);

        config()->set('ebay_legacy_mappings', $this->mappings);
    }

    protected function getProductRepository(): SalesChannelProductRepositoryInterface
    {
        return $this->ebayProductsRepository;
    }

    public function cacheProductListingQuantity(?array $productIds = []): void
    {
        // TODO: Implement cacheProductListingQuantity() method.
    }

    // TODO: Implement getStartDateForNew for legacy products before it gets used

    protected function postCreateSkuProducts()
    {
        // TODO: Implement postCreateSkuProducts() method.
    }

    public function syncInventory(?array $ids = []): void
    {
        if (!$this->integrationInstances->checkSkuIsMasterOfStock($this->integrationInstance)) {
            return;
        }
        $parameters = new EbaySyncInventoryAdt($this->ebayProductsRepository->getProductListingsForInventorySync($ids));
        $this->listings->updateInventorySyncedForIds($this->client->syncInventory($parameters));
    }

    /**
     * @throws Exception
     */
    public function hydrateProductTemplate(Product $product, SalesChannelProductTemplate $template): array
    {
        $fields = $template->fields;
        $hydratedFields = $this->hydrateTemplateFields($fields, $product);

        $fieldsForPayload = $hydratedFields->mapWithKeys(function (SalesChannelHydratedFieldData $field) {
            return [$field->name => $field->value];
        })->toArray();

        return [
            'fields' => $hydratedFields,
            'payload' => $this->buildPayload($fieldsForPayload)
        ];
    }

    /**
     * @throws Exception
     */
    private function hydrateTemplateFields(Collection $fields, Product $product): Collection
    {
        return $fields->map(function (SalesChannelProductTemplateField $field) use ($product) {
            return SalesChannelHydratedFieldData::from([
                'name' => $field->salesChannelFieldDefinition->field_name,
                'value' => match ($field->type) {
                    SalesChannelProductTemplateFieldTypeEnum::STATIC => $field->default_value,
                    SalesChannelProductTemplateFieldTypeEnum::DYNAMIC => app(ProductManager::class)->evaluateTemplate($field->default_value, $product),
                } ?? '',
                'data_type' => $field->salesChannelFieldDefinition->data_type,
                'validation_rules' => $field->salesChannelFieldDefinition->validation_rules,
                'sort_order' => $field->salesChannelFieldDefinition->sort_order,
            ]);
        })->sortBy('sort_order')->values();
    }

    public function buildPayload(array $fields): string
    {
        return View::make(
            "ebay::trading.AddItem",
            $fields
        )->render();
    }

    public function publish(Product $product, string $payload, array $metadata): PublishProductListingResponseData
    {
        $xmlPayload = simplexml_load_string($payload);
        $arrayPayload = json_decode(json_encode($xmlPayload), true);

        // Handle pictures
        if (isset($arrayPayload['Item']['PictureDetails']['PictureURL'])) {
            $uploadedUrls = [];
            foreach (Arr::wrap(@$arrayPayload['Item']['PictureDetails']['PictureURL']) as $pictureUrl)
            {
                $uploadResponse = $this->client->uploadSiteHostedPictures($pictureUrl);
                $uploadedUrls[] = $uploadResponse['SiteHostedPictureDetails']['FullURL'];
            }
            $xmlPayload->Item->PictureDetails = null;
            foreach ($uploadedUrls as $uploadedUrl) {
                $xmlPayload->Item->PictureDetails->addChild('PictureURL', $uploadedUrl);
            }
            $payload = $xmlPayload->asXML();
        }

        $responseArray = $this->client->publish($payload, @$metadata['ebaySiteId']);

        $itemId = $responseArray['ItemID'];
        $responseArrayProduct = $this->client->getProduct($itemId);

        $productCollection = EbayLegacyProductData::collection([$responseArrayProduct['Item']]);

        $this->productRepository->saveForIntegration($this->integrationInstance, $productCollection->toCollection());

        /** @var EbayLegacyProduct $ebayProduct */
        $ebayProduct = $this->productRepository->getProductFromId($this->integrationInstance, $itemId);

        $productListingData = ProductListingDto::from([
            'listing_sku' => $ebayProduct->SKU,
            'sales_channel_listing_id' => $itemId,
            'title' => $ebayProduct->Title,
            'price' => $ebayProduct->CurrentPriceValue,
            'sales_channel_id' => $this->integrationInstance->salesChannel->id,
            'document_id' => $ebayProduct->id,
            'document_type' => EbayLegacyProduct::class,
            'product_id' => $product->id,
            'metadata' => [
                'createResponse' => $responseArray
            ],
        ]);

        return PublishProductListingResponseData::from([
            'productListing' => ProductListing::create($productListingData->toArray()),
            'response' => $responseArray
        ]);
    }

    public function getSuggestedCategories(string $query): array
    {
        $data = collect($this->client->getSuggestedCategories($query));

        return $data->get('SuggestedCategoryArray')['SuggestedCategory'];
    }

    public function relistItem(EbayReviseItemAdt $parameters): array
    {
        return $this->client->relistItem($parameters);
    }

    /**
     * @throws Exception
     */
    private function getSiteIdFromSite(string $site): string
    {
        return match ($site) {
            'eBayMotors' => '100',
            'US' => '0',
            default => throw new Exception('Ebay Site '.$site.' not supported'),
        };
    }

    public function refreshProducts(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        $response = parent::refreshProducts($parameters);
        // Relist auctions after getting latest ebay product data
        $this->relistAuctions();

        return $response;
    }

    public function refreshItem(string $itemId): array
    {
        $responseArrayProduct = $this->client->getProduct($itemId);

        $productCollection = EbayLegacyProductData::collection([$responseArrayProduct['Item']]);

        $this->productRepository->saveForIntegration($this->integrationInstance, $productCollection->toCollection());

        return $responseArrayProduct;
    }

    public function relistAuctions(): void
    {
        $ebayProductsToRelist = $this->ebayProductsRepository->getEbayProductsEligibleForRelist($this->integrationInstance);

        $ebayProductsToRelist->each(/**
         * @throws Exception
         */ function (EbayLegacyProduct $originalEbayProduct) {
           $ebayProductSetting = $originalEbayProduct->productListing->product->ebayProductSettings->where('integration_instance_id', $this->integrationInstance->id)->first();

           $StartPrice = $originalEbayProduct->CurrentPriceValue;
           $StartPrice = $StartPrice - ($StartPrice * $ebayProductSetting->auction_until_sold_price_reduction_percentage / 100);
           if ($ebayProductSetting->auction_until_sold_minimum_price) {
               $StartPrice = max($StartPrice, $ebayProductSetting->auction_until_sold_minimum_price);
           }

           $responseArray = $this->relistItem(new EbayReviseItemAdt(
               ItemID: $originalEbayProduct->ItemID,
               StartPrice: $StartPrice,
               Site: $this->getSiteIdFromSite($originalEbayProduct->json_object['Site']),
           ));

            $itemId = $responseArray['ItemID'];
            $responseArrayProduct = $this->client->getProduct($itemId);

            $productCollection = EbayLegacyProductData::collection([$responseArrayProduct['Item']]);

            $this->productRepository->saveForIntegration($this->integrationInstance, $productCollection->toCollection());

            /** @var EbayLegacyProduct $relistedEbayProduct */
            $relistedEbayProduct = $this->productRepository->getProductFromId($this->integrationInstance, $itemId);

            $productListingData = ProductListingDto::from([
                'listing_sku' => $relistedEbayProduct->SKU,
                'sales_channel_listing_id' => $itemId,
                'title' => $relistedEbayProduct->Title,
                'price' => $relistedEbayProduct->CurrentPriceValue,
                'sales_channel_id' => $this->integrationInstance->salesChannel->id,
                'document_id' => $relistedEbayProduct->id,
                'document_type' => EbayLegacyProduct::class,
                'product_id' => $originalEbayProduct->productListing->product_id,
                'metadata' => [
                    'createResponse' => $responseArray,
                    'originalItemId' => $originalEbayProduct->ItemID,
                    'isRelist' => true,
                ],
            ]);

            ProductListing::create($productListingData->toArray());
        });


    }
}
