<?php

namespace Modules\Ebay\Jobs\Abstractions;

use App\Abstractions\ApiCallInterface;
use App\Models\Concerns\HasJobPayload;
use DateTime;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use InvalidArgumentException;
use Modules\Ebay\Data\EbayResponseData;

abstract class EbayJob implements ShouldQueue, ApiCallInterface
{
    use Dispatchable, HasJobPayload, InteractsWithQueue, Queueable, SerializesModels;

    public ?array $jobPayload = [];
    public int $uniqueFor = 15 * 60;
    public int $timeout = 15 * 60;
    public int $maxExceptions = 1;
    public bool $isPageEnabledJob = false;

    /**
     * Initialization steps for every job.
     *
     * @throws InvalidArgumentException
     */
    protected function jobStarted(): void
    {
        $job = $this->getJobPayload();
        if ($job) {
            $this->jobPayload = $job->payload;
        }
    }

    /**
     * @throws InvalidArgumentException
     */
    protected function jobCompleted(): void
    {
        $this->deleteJobPayload();
    }

    protected function nextPage(): void
    {
        $this->release();
    }

    public function setTimeout($timeout): static
    {
        $this->timeout = $timeout;
        $this->uniqueFor = $this->timeout;
        return $this;
    }

    /**
     * Determine the time at which the job should time out.
     */
    public function retryUntil(): DateTime
    {
        return now()->addSeconds($this->timeout);
    }

    /**
     * Execute the job.
     *
     * @throws Exception
     */
    public function handle(): void
    {
        $this->jobStarted();

        try {
            $ebayResponseDto = $this->apiCall();
            $this->processApiResponse($ebayResponseDto);
        } catch (ConnectionException) {
            $this->handleTimeoutException();
        }
    }

    /**
     * Process the API response.
     *
     * @param EbayResponseData|null $ebayResponseDto
     */
    protected function processApiResponse(?EbayResponseData $ebayResponseDto): void
    {
        if ($ebayResponseDto instanceof EbayResponseData) {
            $this->saveJobPayload($this->getPayloadData($ebayResponseDto));

            if ($this->shouldReleaseNextPage($ebayResponseDto)) {
                $this->nextPage();
            }
        } else {
            $this->jobCompleted();
        }
    }

    /**
     * Get payload data based on the response.
     *
     * @param EbayResponseData $ebayResponseDto
     * @return array
     */
    protected function getPayloadData(EbayResponseData $ebayResponseDto): array
    {
        return $this->isPageEnabledJob
            ? ['nextPage' => $ebayResponseDto->nextPage]
            : ['nextEndpoint' => $ebayResponseDto->nextEndpoint];
    }

    /**
     * Determine if the job should release the next page.
     *
     * @param EbayResponseData $ebayResponseDto
     * @return bool
     */
    protected function shouldReleaseNextPage(EbayResponseData $ebayResponseDto): bool
    {
        if ($this->isPageEnabledJob && $ebayResponseDto->nextPage) {
            Log::debug('Job: Next page ' . $ebayResponseDto->nextPage);
            return true;
        } elseif (!$this->isPageEnabledJob && $ebayResponseDto->nextEndpoint) {
            Log::debug('Job: Next page ' . $ebayResponseDto->nextEndpoint);
            return true;
        }
        return false;
    }

    protected function handleTimeoutException(int $delay = 60): void
    {
        $this->release($delay);
    }
}