<?php

namespace Modules\Ebay\Http\Controllers;

use App\Abstractions\AbstractOauthAuthenticationCallbackCallbackController;
use App\Abstractions\Integrations\AbstractIntegrationInstanceRepository;
use App\Abstractions\Integrations\IntegrationClientInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Response;
use Exception;
use Illuminate\Http\RedirectResponse;
use Modules\Ebay\ApiDataTransferObjects\EbayGetLegacyProductsAdt;
use Modules\Ebay\ApiDataTransferObjects\EbayGetOrdersAdt;
use Modules\Ebay\Data\EbayAuthorizationResponseData;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Jobs\RefreshEbayLegacyProductsJob;
use Modules\Ebay\Jobs\RefreshEbayOrdersJob;
use Modules\Ebay\Repositories\EbayIntegrationInstanceRepository;
use Modules\Ebay\Repositories\EbayLegacyProductRepository;
use Modules\Ebay\Repositories\EbayOrderRepository;
use Modules\Ebay\Services\EbayClient;

class EbayAuthenticationController extends AbstractOauthAuthenticationCallbackCallbackController
{
    public function getIntegrationInstanceRepository(): AbstractIntegrationInstanceRepository
    {
        return app(EbayIntegrationInstanceRepository::class);
    }

    /**
     * @throws Exception
     */
    public function getClient(IntegrationInstanceInterface $integrationInstance): IntegrationClientInterface
    {
        return new EbayClient($integrationInstance);
    }

    public function getStateParameterName(): string
    {
        return 'state';
    }

    /**
     * @throws Exception
     */
    public function getRedirectUrl(EbayIntegrationInstance $integrationInstance): Response
    {
        return $this->response->addData((new EbayClient($integrationInstance))->getRedirectUrl());
    }

    /**
     * @throws Exception
     */
    public function callback(EbayAuthorizationResponseData $data): RedirectResponse
    {
        /** @var EbayIntegrationInstance $integrationInstance */
        $integrationInstance = parent::abstractCallback($data);
        dispatch(new RefreshEbayLegacyProductsJob($integrationInstance, new EbayGetLegacyProductsAdt(
            all: true
        )))->onQueue('sales-channels');
        dispatch(new RefreshEbayOrdersJob($integrationInstance, new EbayGetOrdersAdt(
            created_after: app(EbayOrderRepository::class)->getStartDateForNew($integrationInstance)
        )))->onQueue('sales-channels');

        return redirect('/integrations/ebay/' . $integrationInstance->id . '/dashboard');
    }

    /**
     * @throws Exception
     */
    public function oAuthFailed()
    {
        throw new Exception('Failed to authenticate eBay user.');
    }
}
