<?php

namespace Modules\Ebay\Console;

use Modules\Ebay\Abstractions\AbstractEbayCommand;
use Modules\Ebay\ApiDataTransferObjects\EbayGetOrdersAdt;
use Modules\Ebay\Entities\EbayIntegrationInstance;
use Modules\Ebay\Jobs\RefreshEbayOrdersJob;

class RefreshEbayOrdersCommand extends AbstractEbayCommand
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'ebay:orders:refresh 
        {--integrationInstanceId?} 
        {--lastUpdatedAfter= : Last Updated After } 
        {--lastUpdatedBefore= : Last Updated Before } 
        {--daysUpdated= : Last Updated Within Days Ago }
        {--createdAfter= : Created After } 
        {--createdBefore= : Created Before } 
        {--daysCreated= : Created Within Days Ago }
        {--fulfillmentStatus= : Order Fulfillment Status (NOT_STARTED|IN_PROGRESS|FULFILLED)}
        ';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Get Ebay Orders';

    public function handle(): void
    {
        $this->validateInput();
        /*
         * eBay orders by default gets the last 120 days? of orders
         */
        $parameters = new EbayGetOrdersAdt(
            last_updated_after: $this->option('lastUpdatedAfter'),
            last_updated_before: $this->option('lastUpdatedBefore'),
            daysUpdated: $this->option('daysUpdated'),
            created_after: $this->option('createdAfter'),
            created_before: $this->option('createdBefore'),
            daysCreated: $this->option('daysCreated'),
            fulfillmentStatus: $this->option('fulfillmentStatus'),
        );

        $this->integrationInstances()
            ->each(function (EbayIntegrationInstance $ebayIntegrationInstance) use ($parameters) {
                dispatch(new RefreshEbayOrdersJob($ebayIntegrationInstance, $parameters))->onQueue('sales-channels');
            });
    }

    private function validateInput(): void
    {
        $updatedOptions = [
            'lastUpdatedAfter' => $this->option('lastUpdatedAfter'),
            'lastUpdatedBefore' => $this->option('lastUpdatedBefore'),
            'daysUpdated' => $this->option('daysUpdated'),
        ];

        $createdOptions = [
            'createdAfter' => $this->option('createdAfter'),
            'createdBefore' => $this->option('createdBefore'),
            'daysCreated' => $this->option('daysCreated'),
        ];

        $hasUpdatedOptions = $this->hasOptionsSet($updatedOptions);
        $hasCreatedOptions = $this->hasOptionsSet($createdOptions);

        if ($hasUpdatedOptions && $hasCreatedOptions) {
            $this->error('You can only provide either "updated" options or "created" options, but not both.');
            exit(1);
        }

        if ($updatedOptions['daysUpdated'] && ($updatedOptions['lastUpdatedAfter'] || $updatedOptions['lastUpdatedBefore'])) {
            $this->error('If "daysUpdated" is specified, you cannot provide other "updated" date options.');
            exit(1);
        }

        if ($createdOptions['daysCreated'] && ($createdOptions['createdAfter'] || $createdOptions['createdBefore'])) {
            $this->error('If "daysCreated" is specified, you cannot provide other "created" date options.');
            exit(1);
        }
    }

    private function hasOptionsSet(array $options): bool
    {
        return collect($options)->filter()->count() > 0;
    }
}
