<?php

namespace Modules\Ebay\Console;

use App\Abstractions\Integrations\Data\SalesChannelIntegrationSettingsData;
use App\Data\SalesChannelInventorySettingsData;
use App\Data\SalesChannelMasterOfPriceSettingsData;
use App\Data\SalesChannelPricingSettingsData;
use App\Models\IntegrationInstance;
use App\Repositories\SalesChannelRepository;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Modules\Ebay\Data\EbayConnectionSettingsData;
use Modules\Ebay\Data\StoreEbayIntegrationData;
use Modules\Ebay\Entities\EbayIntegration;
use Modules\Ebay\Repositories\EbayIntegrationInstanceRepository;
use Modules\Ebay\Services\EbayClient;

class InitializeEbayIntegrationInstanceCommand extends Command
{
    protected $signature = 'ebay:initialize
                                {name=Ebay : The name of the integration instance}
                                {store_id=1 : The ID of the store}';

    protected $description = 'Create Ebay integration instance and provide authorization URL';

    protected EbayIntegrationInstanceRepository $integrationInstances;

    protected SalesChannelRepository $salesChannels;

    public function __construct()
    {
        parent::__construct();

        $this->integrationInstances = app(EbayIntegrationInstanceRepository::class);
        $this->salesChannels = app(SalesChannelRepository::class);
    }

    public function handle(): void
    {
        $name = $this->argument('name');
        $storeId = $this->argument('store_id');

        $ebayIntegrationInstanceData = StoreEbayIntegrationData::from([
            'integration_id' => EbayIntegration::sole()->id,
            'name' => $name,
            'connection_settings' => EbayConnectionSettingsData::from([]),
            'integration_settings' => SalesChannelIntegrationSettingsData::from([
                'start_date' => Carbon::now()->subYear()->startOfYear(),
                'store_id' => $storeId,
                'auto_link_products' => false,
                'auto_create_products' => false,
                'sales_nominal_code_id' => null,
                'cogs_nominal_code_id' => null,
                'pricing' => SalesChannelPricingSettingsData::from([
                    'masterOfPrice' => SalesChannelMasterOfPriceSettingsData::from([])
                ]),
                'inventory' => SalesChannelInventorySettingsData::from([]),
                'emailCustomers' => false,
                'proforma_marketplace_cost_percentage' => null,
                'proforma_payment_cost_percentage' => null,
            ]),
        ]);

        $ebayIntegrationInstance = IntegrationInstance::create($ebayIntegrationInstanceData->toArray());

        $this->salesChannels->saveFromIntegrationInstance($ebayIntegrationInstance);

        $ebayClient = new EbayClient($ebayIntegrationInstance);
        $redirectUrl = $ebayClient->getRedirectUrl();

        $this->line("Visit the following URL to complete the setup:\n");
        $this->line($redirectUrl);
    }
}
