<?php

namespace Modules\Ebay\ApiDataTransferObjects;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Data\NoteData;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Repositories\NoteRepository;
use Modules\Ebay\Entities\EbayOrder;
use Modules\Ebay\Entities\EbayOrderItem;
use Modules\Ebay\Exceptions\EbayOriginalSalesOrderLineDeletedException;

class EbayFulfillOrderAdt implements ApiDataTransformerInterface
{

    public function __construct(
        public SalesOrderFulfillment $salesOrderFulfillment,
        public ?string $ebayOrderId = null,
    )
    {}

    public function transform(): array
    {
        $lines = $this->salesOrderFulfillment->salesOrderFulfillmentLines->map(function (SalesOrderFulfillmentLine $line) {
            $salesOrderLine = $line->salesOrderLine;
            if (!$ebayOrderItem = EbayOrderItem::where(EbayOrderItem::getTableUniqueId(), $salesOrderLine->sales_channel_line_id)->first())
            {
                app(NoteRepository::class)->create(NoteData::collection([NoteData::from([
                    'link_id' => $salesOrderLine->salesOrder->id,
                    'link_type' => SalesOrder::class,
                    'user_id' => auth()->user()?->id,
                    'note' => 'Ebay Tracking cannot be submitted because the original sales order line that was mapped to the ebay sales order line has been deleted.',
                ])]));
                throw new EbayOriginalSalesOrderLineDeletedException('Ebay Tracking for ' . $salesOrderLine->salesOrder->sales_order_number . ' cannot be submitted because the original sales order line that was mapped to the ebay sales order line has been deleted.');
            };
            return [
                'lineItemId' => $ebayOrderItem->{EbayOrderItem::getTableUniqueId()},
                'quantity' => $line->quantity,
            ];
        })->toArray();

        /** @var EbayOrder $ebayOrder */
        $ebayOrder = $this->salesOrderFulfillment->salesOrder->salesChannelOrder;
        $this->ebayOrderId = $ebayOrder->orderId;

        return [
            'lineItems' => $lines,
            'shippedDate' => $this->salesOrderFulfillment->fulfilled_at->toIso8601String(),
            'shippingCarrierCode' => $this->salesOrderFulfillment->fulfilledShippingMethod?->name ?? $this->salesOrderFulfillment->fulfilled_shipping_method,
            'trackingNumber' => $this->salesOrderFulfillment->tracking_number,
        ];
    }
}