<?php

namespace Modules\Amazon\Tests\Unit;

use Illuminate\Support\Facades\Artisan;
use Modules\Amazon\Actions\RequestAmazonReport;
use Modules\Amazon\Data\RequestAmazonReportData;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Entities\AmazonReportRequest;
use Modules\Amazon\Enums\Entities\AmazonFeedProcessingStatusEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class AmazonReportOverlapTest extends TestCase
{
    use FastRefreshDatabase;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    protected function setUp(): void
    {
        parent::setUp();
        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()->create();
    }

    /**
     * @throws Throwable
     */
    public function test_it_cant_create_a_report_already_created_and_incomplete(): void
    {
        $amazonReports = AmazonReport::factory(2)
            ->sequence(
                [
                    'integration_instance_id' => $this->amazonIntegrationInstance->id,
                    'reportType' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS,
                    'processingStatus' => AmazonFeedProcessingStatusEnum::DONE,
                    'dataStartTime' => '2021-01-01',
                    'dataEndTime' => '2021-01-31 23:59:59',
                ],
                [
                    'integration_instance_id' => $this->amazonIntegrationInstance->id,
                    'reportType' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS,
                    'processingStatus' => AmazonFeedProcessingStatusEnum::IN_PROGRESS,
                    'dataStartTime' => '2021-02-01',
                    'dataEndTime' => '2021-02-05 23:59:59',
                ],
            )
            ->create();

        $inProgressReport = $amazonReports->filter(function ($report) {
            return $report->processingStatus->value === AmazonFeedProcessingStatusEnum::IN_PROGRESS->value;
        })->first();

        $reports = (new RequestAmazonReport(RequestAmazonReportData::from([
            'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
            'report_type' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS,
            'createImmediately' => false,
        ])))->handle();

        $this->assertEmpty($reports->createdRequests);

        /*
        |--------------------------------------------------------------------------
        | Test it can request a report already requested and completed
        |--------------------------------------------------------------------------
        */

        $inProgressReport->update(['processingStatus' => AmazonFeedProcessingStatusEnum::DONE]);

        $reports = (new RequestAmazonReport(RequestAmazonReportData::from([
            'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
            'report_type' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS,
            'createImmediately' => false,
        ])))->handle();

        $this->assertNotEmpty($reports->createdRequests);

        // Assert it will choose the correct start date for continuation of report type
        $this->assertEquals('2021-02-06 00:00:00', $reports->createdRequests[0]->dataStartTime);
    }

    public function test_it_can_create_multiple_report_requests_for_missing_gaps_in_reports_when_requested(): void
    {
        AmazonReport::factory(3)
            ->sequence(
                [
                    'integration_instance_id' => $this->amazonIntegrationInstance->id,
                    'reportType' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS,
                    'processingStatus' => AmazonFeedProcessingStatusEnum::DONE,
                    'dataStartTime' => '2021-01-01',
                    'dataEndTime' => '2021-01-31 23:59:59',
                ],
                [
                    'integration_instance_id' => $this->amazonIntegrationInstance->id,
                    'reportType' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS,
                    'processingStatus' => AmazonFeedProcessingStatusEnum::DONE,
                    'dataStartTime' => '2021-03-01',
                    'dataEndTime' => '2021-03-31 23:59:59',
                ],
                [
                    'integration_instance_id' => $this->amazonIntegrationInstance->id,
                    'reportType' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS,
                    'processingStatus' => AmazonFeedProcessingStatusEnum::DONE,
                    'dataStartTime' => '2021-05-01',
                    'dataEndTime' => '2021-05-31 23:59:59',
                ],
            )
            ->create();

        Artisan::call('amazon:reports:request ' . $this->amazonIntegrationInstance->id . ' ' . AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS->value . ' --gap');

        $this->assertDatabaseHas(AmazonReportRequest::class, [
            'dataStartTime' => '2021-02-01 00:00:00',
            'dataEndTime' => '2021-02-28 23:59:59',
        ]);

        $this->assertDatabaseHas(AmazonReportRequest::class, [
            'dataStartTime' => '2021-04-01 00:00:00',
            'dataEndTime' => '2021-04-30 23:59:59',
        ]);
    }
}
