<?php

namespace Modules\Amazon\Tests\Feature\Managers;

use App\Exceptions\DropshipWithOpenOrdersException;
use App\Models\Address;
use App\Models\Currency;
use App\Models\Customer;
use App\Models\FinancialLine;
use App\Models\InventoryMovement;
use App\Models\OrderLink;
use App\Models\Payment;
use App\Models\PaymentType;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\ShippingMethod;
use App\Models\Warehouse;
use App\Repositories\SalesOrderFulfillmentRepository;
use Exception;
use Http;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Actions\CreateAmazonFeed;
use Modules\Amazon\Actions\InitializeFbaWarehouse;
use Modules\Amazon\Entities\AmazonFbaReportInventory;
use Modules\Amazon\Entities\AmazonFeedRequest;
use Modules\Amazon\Entities\AmazonFeedSubmission;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Entities\AmazonOrderItem;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Enums\Entities\AmazonFulfillmentChannelEnum;
use Modules\Amazon\Enums\Entities\OrderStatusEnum;
use Modules\Amazon\Managers\AmazonFeedManager;
use Modules\Amazon\Managers\AmazonIntegrationInstanceManager;
use Modules\Amazon\Managers\AmazonOrderManager;
use Modules\Amazon\Tests\Feature\Managers\Helpers\AmazonMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Queue;
use Tests\TestCase;
use Throwable;

class AmazonOrderManagerTest extends TestCase
{
    use AmazonMockRequests;
    use FastRefreshDatabase;
    use WithFaker;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    private AmazonOrderManager $amazonOrderManager;

    /**
     * @throws DropshipWithOpenOrdersException
     */
    protected function setUp(): void
    {
        parent::setUp();

        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()
            ->hasSalesChannel()
            ->create();

        (new InitializeFbaWarehouse($this->amazonIntegrationInstance))->handle();
        $this->amazonIntegrationInstance = $this->amazonIntegrationInstance->refresh();

        PaymentType::create([
            'name' => PaymentType::PAYMENT_TYPE_AMAZON,
        ]);

        $this->amazonOrderManager = new AmazonOrderManager($this->amazonIntegrationInstance);

        Http::preventStrayRequests();
    }

    /**
     * Mock get orders API.
     */
    public function mockApiCalls(): void
    {
        $this->mockRefreshOrders();
    }

    public function assertCreateSkuOrders(int $numberOfRecords, int $existingAddressNumberOfRecords, bool $isCustomerExist = true, $isAddressExists = true)
    {
        $amazonOrder = AmazonOrder::with('orderItems')->first();
        $customer = null;

        if ($isCustomerExist && ! is_null($amazonOrder->BuyerInfo)) {
            $query = Customer::query()
                ->where('email', $amazonOrder->BuyerInfo['BuyerEmail']);

            if (isset($amazonOrder->BuyerInfo['BuyerName'])) { // Buyer Name is not provided for AFN orders
                $query->where('name', $amazonOrder->BuyerInfo['BuyerName']);
            }
            $customer = $query->first();
        }

        $this->assertDatabaseCount((new SalesOrder())->getTable(), $numberOfRecords);
        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'sales_channel_id' => $amazonOrder->integrationInstance->salesChannel->id,
            'sales_order_number' => $amazonOrder->AmazonOrderId,
            'currency_id' => Currency::whereCode($amazonOrder->OrderCurrency)->first()->id,
            'order_date' => $amazonOrder->PurchaseDateUtc,
            'fulfillment_channel' => $amazonOrder->FulfillmentChannel,
            'fulfilled_at' => $amazonOrder->OrderStatus->value == OrderStatusEnum::STATUS_SHIPPED() ? $amazonOrder->LastUpdateDateUtc : null,
            'ship_by_date' => $amazonOrder->LatestShipDateUtc ?? null,
            'deliver_by_date' => $amazonOrder->LatestDeliveryDateUtc ?? null,
            'shipping_method_id' => null, // AFN / (Carrier ID - MFN)
            'requested_shipping_method' => $amazonOrder->ShipServiceLevel,
            // TODO: Jatin please make this work
            //'is_fba' => $amazonOrder->FulfillmentChannel == 'AFN',
            'customer_id' => $customer ? $customer->id : null,
            'shipping_address_id' => $customer ? $customer->default_shipping_address_id : null,
            'billing_address_id' => $customer ? $customer->default_billing_address_id : null,
            // 'order_status' => $amazonOrder->FulfillmentChannel == 'MFN' ? SalesOrder::STATUS_OPEN : SalesOrder::STATUS_CLOSED,
            'sales_channel_order_type' => AmazonOrder::class,
            'sales_channel_order_id' => $amazonOrder->id,
        ]);

        $this->assertDatabaseCount((new SalesOrderLine())->getTable(), $numberOfRecords);
        $this->assertDatabaseCount((new FinancialLine())->getTable(), $numberOfRecords);

        if ($isCustomerExist && ! is_null($amazonOrder->BuyerInfo)) {
            $this->assertDatabaseCount((new Customer())->getTable(), $numberOfRecords);
            $this->assertDatabaseHas((new Customer())->getTable(), [
                // 'name' => encryptValue($amazonOrder->BuyerInfo['BuyerName']), // Buyer Name is not provided for AFN orders
                'email' => $amazonOrder->BuyerInfo['BuyerEmail'],
                // 'zip' => @$amazonOrder->ShippingAddress['PostalCode'],
                // 'address1' => @$amazonOrder->ShippingAddress['AddressLine1'],
                // 'company' => null,
                // 'fax' => null,
            ]);

            if ($isAddressExists) {
                $this->assertDatabaseCount((new Address())->getTable(), $existingAddressNumberOfRecords + $numberOfRecords);
            }
        }
    }

    /**
     * Customer with address.
     *
     * @throws Exception
     * @throws Throwable
     */
    public function test_create_sku_orders_with_customers_with_address_with_bulk(): void
    {
        //Set number of records
        $numberOfRecords = SalesOrder::BULK_THRESHOLD + 1;
        $existingAddressNumberOfRecords = Address::query()->count();

        //Create Amazon Orders
        AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance, 'integrationInstance')
            ->count($numberOfRecords)
            ->has(AmazonOrderItem::factory(), 'orderItems')
            ->create();

        //Create SKU Orders
        $this->amazonOrderManager->createSkuOrders([], true);
        $this->assertCreateSkuOrders($numberOfRecords, $existingAddressNumberOfRecords);
    }

    /**
     * Customer with address.
     *
     * @throws Exception
     */
    public function test_create_sku_orders_without_customer_without_address_with_bulk(): void
    {
        //Set number of records
        $numberOfRecords = SalesOrder::BULK_THRESHOLD + 1;
        $existingAddressNumberOfRecords = Address::query()->count();

        //Create Amazon Orders
        AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance)
            ->count($numberOfRecords)
            ->has(AmazonOrderItem::factory(), 'orderItems')
            ->make()
            ->map(function ($amazonOrder) {
                $amazonOrder->json_object = array_merge($amazonOrder->json_object, [
                    'BuyerInfo' => null,
                ]);
                $amazonOrder->save();

                AmazonOrderItem::factory()->for($amazonOrder, 'order')->create();

                return $amazonOrder;
            });

        //Create SKU Orders
        $this->amazonOrderManager->createSkuOrders([], true);

        $this->assertCreateSkuOrders($numberOfRecords, $existingAddressNumberOfRecords, false, false);
    }

    /**
     * Customer without address.
     *
     * @throws Exception
     */
    public function test_create_sku_orders_with_customer_without_address_with_bulk(): void
    {
        //Set number of records
        $numberOfRecords = SalesOrder::BULK_THRESHOLD + 1;
        $existingAddressNumberOfRecords = Address::query()->count();

        //Create Amazon Orders
        AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance)
            ->count($numberOfRecords)
            ->has(AmazonOrderItem::factory(), 'orderItems')
            ->make()
            ->map(function ($amazonOrder) {
                $amazonOrder->json_object = array_merge($amazonOrder->json_object, [
                    'ShippingAddress' => null,
                    'OrderStatus' => collect(OrderStatusEnum::STATUSES_ACTIVE)->first()->value,
                ]);
                $amazonOrder->save();

                AmazonOrderItem::factory()->for($amazonOrder, 'order')->create();

                return $amazonOrder;
            });

        //Create SKU Orders
        $this->amazonOrderManager->createSkuOrders([], true);

        $this->assertCreateSkuOrders($numberOfRecords, $existingAddressNumberOfRecords, true, false);
        $this->assertDatabaseCount((new Payment())->getTable(), $numberOfRecords);
    }

    /**
     * @throws Exception
     */
    public function test_create_sku_orders_with_customer_with_address(): void
    {
        Queue::fake();
        //Set number of records
        $existingAddressNumberOfRecords = Address::query()->count();
        $numberOfRecords = SalesOrder::BULK_THRESHOLD - 1;

        //Create Amazon Orders
        AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance, 'integrationInstance')
            ->count($numberOfRecords)
            ->has(AmazonOrderItem::factory()->state(function (array $attributes) {
                return [
                    'json_object->ShippingPrice->Amount' => '3.00',
                ];
            }), 'orderItems')
            ->create();

        //Create SKU Orders
        $this->amazonOrderManager->createSkuOrders([], true);

        $this->assertCreateSkuOrders($numberOfRecords, $existingAddressNumberOfRecords);
    }

    /**
     * @throws Throwable
     */
    public function test_create_sku_orders_with_customer_without_address(): void
    {
        Queue::fake();
        //Set number of records
        $existingAddressNumberOfRecords = Address::query()->count();
        $numberOfRecords = SalesOrder::BULK_THRESHOLD - 1;

        //Create Amazon Orders
        AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance)
            ->count($numberOfRecords)
            ->has(AmazonOrderItem::factory()->state(function (array $attributes) {
                return [
                    'json_object->ShippingPrice->Amount' => '3.00',
                ];
            }), 'orderItems')
            ->make()
            ->map(function ($amazonOrder) {
                $amazonOrder->json_object = array_merge($amazonOrder->json_object, [
                    'ShippingAddress' => null,
                ]);
                $amazonOrder->save();

                AmazonOrderItem::factory()->for($amazonOrder, 'order')->create([
                    'json_object->ShippingPrice->Amount' => '3.00',
                ]);

                return $amazonOrder;
            });

        //Create SKU Orders
        $this->amazonOrderManager->createSkuOrders([], true);

        $this->assertCreateSkuOrders($numberOfRecords, $existingAddressNumberOfRecords, true, false);
    }

    /**
     * @throws Exception
     */
    public function test_create_sku_orders_without_customer_without_address(): void
    {
        Queue::fake();
        //Set number of records
        $existingAddressNumberOfRecords = Address::query()->count();
        $numberOfRecords = SalesOrder::BULK_THRESHOLD - 1;

        //Create Amazon Orders
        AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance)
            ->count($numberOfRecords)
            ->has(AmazonOrderItem::factory(), 'orderItems')
            ->make()
            ->map(function ($amazonOrder) {
                $amazonOrder->json_object = array_merge($amazonOrder->json_object, [
                    'BuyerInfo' => null,
                    'ShippingAddress' => null,
                    'OrderStatus' => collect(OrderStatusEnum::STATUSES_ACTIVE)->first()->value,
                ]);
                $amazonOrder->save();

                AmazonOrderItem::factory()->for($amazonOrder, 'order')->create([
                    'json_object->ShippingPrice->Amount' => '3.00',
                ]);

                return $amazonOrder;
            });

        //Create SKU Orders
        $this->amazonOrderManager->createSkuOrders([], true);

        $this->assertCreateSkuOrders($numberOfRecords, $existingAddressNumberOfRecords, false, false);
        $this->assertDatabaseCount((new Payment())->getTable(), $numberOfRecords);
    }

    public function test_fulfill_order_for_sales_order_fulfillment()
    {
        $numberOfRecordsToTest = 10;

        //Create Amazon Orders
        AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance)
            ->count($numberOfRecordsToTest)
            ->has(AmazonOrderItem::factory(), 'orderItems')
            ->make()
            ->map(function ($amazonOrder) {
                $amazonOrder->json_object = array_merge($amazonOrder->json_object, [
                    'BuyerInfo' => null,
                    'ShippingAddress' => null,
                    'FulfillmentChannel' => AmazonFulfillmentChannelEnum::MFN,
                ]);
                $amazonOrder->save();

                AmazonOrderItem::factory()->for($amazonOrder, 'order')->create([
                    'json_object->ShippingPrice->Amount' => '3.00',
                ]);

                return $amazonOrder;
            });

        //Create SKU Orders
        $this->amazonOrderManager->createSkuOrders([], true);

        $shippingMethod = ShippingMethod::query()->first();

        //If no orders are there yet then we don't create feed request
        $this->amazonOrderManager->fulfillPendingAmazonMfnOrders();
        $this->assertDatabaseEmpty((new AmazonFeedRequest())->getTable());

        SalesOrder::all()->each(function ($salesOrder) use ($shippingMethod) {
            SalesOrderFulfillmentLine::factory()
                ->for(
                    SalesOrderFulfillment::factory()
                        ->for($salesOrder)
                        ->for($shippingMethod, 'fulfilledShippingMethod')
                        ->create([
                            'tracking_number' => $this->faker->sha1,
                        ])
                )
                ->create();
        });

        $this->amazonOrderManager->fulfillPendingAmazonMfnOrders();
        (new CreateAmazonFeed(AmazonFeedRequest::query()->first()))->handle();

        $this->assertDatabaseCount((new AmazonFeedRequest())->getTable(), 0);
        $this->assertDatabaseCount((new AmazonFeedSubmission())->getTable(), 1);

        //Feed submission process starts
        $amazonFeesSubmission = AmazonFeedSubmission::query()->first();

        //Check orders
        $fulfilledSalesOrders = $amazonFeesSubmission->salesOrderFulfillments()->count();
        $this->assertEquals($numberOfRecordsToTest, $fulfilledSalesOrders);

        //Get Amazon Feed details
        $this->mockRefreshFeed($amazonFeesSubmission);
        (new AmazonFeedManager($this->amazonIntegrationInstance))->getAmazonFeedDetails(
            $amazonFeesSubmission
        );

        //Check orders
        $fulfilledSalesOrders = $amazonFeesSubmission->salesOrderFulfillments()->count();
        $this->assertEquals(10, $fulfilledSalesOrders);

        //Get result feed
        (new AmazonFeedManager($this->amazonIntegrationInstance))->getAmazonFeedResultDocument(
            $amazonFeesSubmission
        );
        $this->assertEquals(2, (new SalesOrderFulfillmentRepository())->getPendingSalesOrderFulfillments($this->amazonIntegrationInstance)->count());
    }

    public function test_amazon_handle_cancel_sales_orders()
    {
        //Create Amazon Orders
        $amazonOrder = AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance)
            ->hasOrderItems()
            ->create()
            ->refresh();
        $amazonOrder->json_object = array_merge($amazonOrder->json_object, [
            'OrderStatus' => OrderStatusEnum::STATUS_SHIPPED->value,
            'FulfillmentChannel' => AmazonFulfillmentChannelEnum::MFN,
        ]);
        $amazonOrder->update();

        //Create SKU orders
        $this->amazonOrderManager->createSkuOrders([], true);
        $amazonOrder->load('salesOrder');

        //Amazon order canceled
        $amazonOrder->json_object = array_merge($amazonOrder->json_object, [
            'OrderStatus' => OrderStatusEnum::STATUS_CANCELED->value,
        ]);
        $amazonOrder->update();

        //Order should not be canceled
        $this->assertTrue(is_null($amazonOrder->salesOrder->canceled_at));

        //Amazon Order is canceled
        $this->amazonOrderManager->handleCancelSalesOrders(collect([
            [
                'AmazonOrderId' => $amazonOrder->AmazonOrderId,
            ],
        ]));
        $amazonOrder->salesOrder->refresh();

        //Sales order must be canceled
        $this->assertFalse(is_null($amazonOrder->salesOrder->canceled_at));
    }

    /**
     * @throws Throwable
     */
    public function test_it_wont_reserve_inventory_for_amazon_mfn_order_for_amazon_fba_warehouse(): void
    {
        Setting::where('key', Setting::KEY_WAREHOUSE_IGNORE_AMAZON_FBA_TYPE)->update([
            'value' => false,
        ]);

        $amazonOrder = AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance)
            ->hasOrderItems()
            ->create()
            ->refresh();
        $amazonOrder->json_object = array_merge($amazonOrder->json_object, [
            'OrderStatus' => OrderStatusEnum::STATUS_UNSHIPPED->value,
            'FulfillmentChannel' => AmazonFulfillmentChannelEnum::MFN,
        ]);
        $amazonOrder->update();

        $orderItem = $amazonOrder->orderItems->first();
        $amazonProduct = $orderItem->product;
        $product = Product::factory()->create();
        $warehouse = $this->amazonIntegrationInstance->warehouse;

        $amazonFbaReportInventory = AmazonFbaReportInventory::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
        ]);

        $jsonObject = $amazonFbaReportInventory->json_object;
        $jsonObject['sku'] = $orderItem->SellerSKU;
        $jsonObject['condition'] = 'New';
        $jsonObject['afn_fulfillable_quantity'] = 10;
        $amazonFbaReportInventory->json_object = $jsonObject;
        $amazonFbaReportInventory->save();

        ProductListing::factory()->create([
            'listing_sku' => $orderItem->SellerSKU,
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'product_id' => $product->id,
            'document_id' => $amazonProduct->id,
            'document_type' => AmazonProduct::class,
        ]);

        (new AmazonOrderManager($this->amazonIntegrationInstance))->createSkuOrders([], true);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'warehouse_id' => $warehouse->id,
        ]);

        // Orders using MFN should not reserve inventory
        $this->assertDatabaseMissing((new InventoryMovement())->getTable(), [
            'link_type' => SalesOrderLine::class,
        ]);
    }

    /**
     * @throws Throwable
     */
    public function test_it_wont_choose_amazon_fba_warehouse_for_mfn_order_based_on_products_inventory(): void
    {
        Setting::where('key', Setting::KEY_WAREHOUSE_IGNORE_AMAZON_FBA_TYPE)->update([
            'value' => false,
        ]);

        $amazonOrder = AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance)
            ->hasOrderItems()
            ->create()
            ->refresh();
        $amazonOrder->json_object = array_merge($amazonOrder->json_object, [
            'OrderStatus' => OrderStatusEnum::STATUS_UNSHIPPED->value,
            'FulfillmentChannel' => AmazonFulfillmentChannelEnum::MFN,
        ]);
        $amazonOrder->update();

        $orderItem = $amazonOrder->orderItems->first();
        $amazonProduct = $orderItem->product;
        $product = Product::factory()->create();
        $warehouse = $this->amazonIntegrationInstance->warehouse;

        $product->setInitialInventory($warehouse->id, 10, 5.00);

        ProductListing::factory()->create([
            'listing_sku' => $orderItem->SellerSKU,
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'product_id' => $product->id,
            'document_id' => $amazonProduct->id,
            'document_type' => AmazonProduct::class,
        ]);

        (new AmazonOrderManager($this->amazonIntegrationInstance))->createSkuOrders([], true);

        $this->assertDatabaseMissing((new SalesOrderLine())->getTable(), [
            'warehouse_id' => $warehouse->id,
        ]);
    }

    /**
     * @throws Throwable
     */
    public function test_it_wont_create_sku_order_for_non_amazon_amazon_order(): void
    {
        $amazonOrder = AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance)
            ->hasOrderItems()
            ->create()
            ->refresh();

        $amazonOrder->json_object = array_merge($amazonOrder->json_object, [
            'OrderStatus' => OrderStatusEnum::STATUS_SHIPPED->value,
            'SalesChannel' => 'Non-Amazon US',
        ]);
        $amazonOrder->update();

        $this->amazonOrderManager->createSkuOrders([], true);

        $this->assertDatabaseEmpty((new SalesOrder())->getTable());
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_create_replacement_order_where_original_sales_order_exists(): void
    {
        //Create Amazon Orders

        AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance, 'integrationInstance')
            ->has(AmazonOrderItem::factory(), 'orderItems')
            ->create([
                'json_object->AmazonOrderId' => 'parentOrder',
            ]);

        AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance, 'integrationInstance')
            ->has(AmazonOrderItem::factory(), 'orderItems')
            ->create([
                'json_object->AmazonOrderId' => 'replacedOrder',
                'json_object->ReplacedOrderId' => 'parentOrder',
                'json_object->IsReplacementOrder' => 'true',
            ]);

        $this->amazonOrderManager->createSkuOrders([], true);

        $originalOrder = SalesOrder::where('sales_order_number', 'parentOrder')->first();
        $replacedOrder = SalesOrder::where('sales_order_number', 'replacedOrder')->first();

        $this->assertDatabaseHas(SalesOrder::class, [
            'id' => $replacedOrder->id,
            'sales_order_number' => 'replacedOrder',
            'is_replacement_order' => true,
        ]);

        $this->assertDatabaseHas(OrderLink::class, [
            'parent_id' => $originalOrder->id,
            'parent_type' => SalesOrder::class,
            'child_id' => $replacedOrder->id,
            'child_type' => SalesOrder::class,
            'link_type' => OrderLink::LINK_TYPE_RESEND,
        ]);
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_create_replacement_order_where_original_sales_order_does_not_exist(): void
    {
        AmazonOrder::factory()
            ->for($this->amazonIntegrationInstance, 'integrationInstance')
            ->has(AmazonOrderItem::factory(), 'orderItems')
            ->create([
                'json_object->AmazonOrderId' => 'replacedOrder',
                'json_object->ReplacedOrderId' => 'parentOrder',
                'json_object->IsReplacementOrder' => 'true',
            ]);

        $this->amazonOrderManager->createSkuOrders([], true);

        $replacedOrder = SalesOrder::where('sales_order_number', 'replacedOrder')->first();

        $this->assertDatabaseHas(SalesOrder::class, [
            'id' => $replacedOrder->id,
            'sales_order_number' => 'replacedOrder',
            'is_replacement_order' => true,
        ]);
    }
}
