<?php

namespace Feature\Managers;

use App\Exceptions\DropshipWithOpenOrdersException;
use App\Http\Requests\StoreInventoryAdjustment;
use App\Models\InventoryMovement;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\PurchaseOrder;
use App\Models\User;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferShipmentReceiptLine;
use Carbon\Carbon;
use Exception;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\Amazon\Actions\InitializeFbaWarehouse;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonFbaInboundShipmentItem;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFbaInboundShipFromMapping;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipment;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipmentItem;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Enums\Entities\AmazonNewFbaInboundShipmentStatusEnum;
use Modules\Amazon\Enums\Entities\AmazonProductFulfillmentChannelEnum;
use Modules\Amazon\Enums\Entities\FbaInboundShipmentStatusEnum;
use Modules\Amazon\Enums\Entities\FbaInventoryLedgerReportEventTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Managers\AmazonFnskuProductManager;
use Modules\Amazon\Managers\AmazonInboundManager;
use Modules\Amazon\Managers\AmazonLedgerManager;
use Modules\Amazon\Managers\AmazonNewInboundManager;
use Modules\Amazon\Tests\AmazonMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class AmazonNewInboundManagerTest extends TestCase
{
    use AmazonMockRequests;
    use FastRefreshDatabase;
    use WithFaker;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    /**
     * @throws DropshipWithOpenOrdersException
     */
    public function setUp(): void
    {
        parent::setUp();

        Queue::fake();

        $this->mockGetAccessToken();

        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()->hasSalesChannel()->create();
        (new InitializeFbaWarehouse($this->amazonIntegrationInstance))->handle();
        $this->amazonIntegrationInstance->refresh();
    }

    /**
     * @throws Throwable
     */
    public function test_it_can_create_warehouse_transfer_from_new_inbound(): void
    {
        Queue::fake();

        Sanctum::actingAs(User::factory()->create());

        /** @var Warehouse $sourceWarehouse */
        $sourceWarehouse = Warehouse::query()->first();

        /** @var AmazonProduct $amazonProduct */
        $amazonProduct = AmazonProduct::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'json_object' => [
                'seller_sku' => 'SKU1',
                'asin1' => 'ASIN1',
                'fulfillment_channel' => AmazonProductFulfillmentChannelEnum::AMAZON_NA,
            ],
        ])
            ->refresh();

        /** @var Product $product */
        $product = Product::factory()->create([
            'sku' => 'SKU1',
        ]);

        $this->postJson('/api/inventory-adjustments', [
            'product_id' => $product->id,
            'adjustment_date' => Carbon::now()->format('Y-m-d'),
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'unit_cost' => 2,
            'warehouse_id' => $sourceWarehouse->id,
            'quantity' => 5,
            'reason' => 'Test',
        ])->assertOk();

        ProductListing::factory()->create([
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'listing_sku' => 'SKU1',
            'document_id' => $amazonProduct->id,
            'document_type' => AmazonProduct::class,
            'product_id' => $product->id,
        ]);

        $amazonInboundShipment = AmazonNewFbaInboundShipment::factory()->create([
                'integration_instance_id' => $this->amazonIntegrationInstance->id,
                'json_object' => [
                    'shipmentConfirmationId' => 'FBA16TDCRLV9',
                    'status' => 'WORKING',
                    'source' => [
                        'address' => [
                            'name' => $sourceWarehouse->name,
                        ]
                    ],
                    'destination' => [
                        'address' => [
                            'name' => $this->amazonIntegrationInstance->warehouse->name,
                            'countryCode' => 'US'
                        ]
                    ],
                ],
            ])
            ->refresh();

        AmazonNewFbaInboundShipmentItem::factory()->create([
            'amazon_new_fba_inbound_shipment_id' => $amazonInboundShipment->id,
            'json_object' => [
                'msku' => 'SKU1',
                'fnsku' => 'FNSKU1',
                'asin' => 'ASIN1',
                'quantity' => 3,
            ],
        ]);

        AmazonFbaInboundShipFromMapping::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'name' => $sourceWarehouse->name,
            'link_type' => Warehouse::class,
            'link_id' => $sourceWarehouse->id,
        ]);

        /** @var AmazonNewFbaInboundShipmentItem $amazonFbaInboundShipmentItem */
        $amazonFbaInboundShipmentItem = AmazonNewFbaInboundShipmentItem::query()->first();

        (new AmazonFnskuProductManager($this->amazonIntegrationInstance))->generateFnskuProducts();

        $this->assertDatabaseHas((new AmazonFnskuProduct())->getTable(), [
            'fnsku' => $amazonFbaInboundShipmentItem->fnsku,
            'product_id' => $product->id,
        ]);

        $response = $this->getJson(route('amazon.new-inbound.shipments.get-unlinked', $this->amazonIntegrationInstance->id));

        $response->assertOk()
            ->assertJsonStructure([
                'data' => [
                    '*' => [
                        'id',
                        'shipmentConfirmationId',
                    ],
                ],
            ]);

        $amazonFbaInboundShipmentId = $response->json('data.0.id');

        (new AmazonNewInboundManager($this->amazonIntegrationInstance))->process([$amazonFbaInboundShipmentId]);

        $this->assertEquals(
            1,
            WarehouseTransfer::query()
                ->where('from_warehouse_id', $sourceWarehouse->id)
                ->where('to_warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
                ->where('warehouse_transfer_number', 'FBA16TDCRLV9')
                ->where('transfer_status', WarehouseTransfer::TRANSFER_STATUS_OPEN)
                ->count()
        );

        $this->assertEquals(
            1,
            AmazonNewFbaInboundShipment::query()
                ->whereHas('skuLink', function ($query) {
                    $query->where('warehouse_transfer_number', 'FBA16TDCRLV9');
                })
                ->count()
        );

        $this->assertEquals(
            1,
            InventoryMovement::query()
                ->where('warehouse_id', $sourceWarehouse->id)
                ->where('quantity', -3)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                ->count()
        );

        $this->assertEquals(
            0,
            InventoryMovement::query()
                ->where('warehouse_id', $sourceWarehouse->id)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                ->sum('quantity')
        );

        $this->assertEquals(
            1,
            InventoryMovement::query()
                ->where('warehouse_id', $this->amazonIntegrationInstance->warehouse->id)
                ->where('quantity', 3)
                ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_IN_TRANSIT)
                ->count()
        );

        WarehouseTransfer::query()->first()->delete();

        $this->assertEquals(
            0,
            AmazonNewFbaInboundShipment::query()
                ->whereNull('sku_link_id')
                ->whereNull('sku_link_type')
                ->count(), 'AmazonFbaInboundShipment should clear the relationship when the warehouse transfer is deleted');
    }

    /**
     * @throws Exception
     */
    public function test_it_can_delete_sku_link_for_cancelled_inbound()
    {
        $inbound = AmazonNewFbaInboundShipment::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
        ]);

        $purchaseOrder = PurchaseOrder::factory()->create([
            'purchase_order_number' => $inbound->shipmentConfirmationId,
        ]);

        $jsonObject = $inbound->json_object;
        $jsonObject['status'] = AmazonNewFbaInboundShipmentStatusEnum::CANCELLED;
        $inbound->update([
            'sku_link_id' => $purchaseOrder->id,
            'sku_link_type' => PurchaseOrder::class,
            'json_object' => $jsonObject
        ]);

        $manager = new AmazonNewInboundManager($this->amazonIntegrationInstance);
        $manager->processCancellations();

        $this->assertDatabaseEmpty((new PurchaseOrder())->getTable());
    }

    /*
     * TODO: Pending tests
     *  Test that when a ledger report comes in with eventType = "Receipts" that the warehouse transfer created gets
     *  received and the expected inventory movements occur and the relationship between the Ledger and the Warehouse
     *  Transfer Shipment Receipt Line exists (via sku_link in the ledgers table).
     */
}
