<?php

namespace Feature\Controllers;

use App\Models\Address;
use App\Models\Integration;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use App\Models\ShippingCarrier;
use App\Models\ShippingMethod;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\SalesOrder\SalesOrderManager;
use Cassandra\Set;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\Amazon\Actions\InitializeFbaShippingMethods;
use Modules\Amazon\Entities\AmazonFbaReportInventory;
use Modules\Amazon\Entities\AmazonFulfillmentOrder;
use Modules\Amazon\Entities\AmazonFulfillmentOrderItem;
use Modules\Amazon\Entities\AmazonFulfillmentShipment;
use Modules\Amazon\Entities\AmazonFulfillmentShipmentItem;
use Modules\Amazon\Entities\AmazonFulfillmentShipmentPackage;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Enums\Entities\FulfillmentShippingSpeedCategoryEnum;
use Modules\Amazon\Tests\AmazonMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class AmazonOutboundFulfillmentControllerTest extends TestCase
{
    use AmazonMockRequests;
    use FastRefreshDatabase;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();
        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->has(Warehouse::factory(1, ['type' => Warehouse::TYPE_AMAZON_FBA]))
            ->create([
                'integration_id' => Integration::where('name', 'Amazon US')->first()->id,
            ]);

        $this->mockGetAccessToken();
        $this->mockGetRestrictedDataToken();

        (new InitializeFbaShippingMethods($this->amazonIntegrationInstance))->handle();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Throwable
     */
    public function test_amazon_outbound_fulfillment_controller(): void
    {
        $this->mockGetFulfillmentOrder();
        $this->mockListAllFulfillmentOrders();

        /*
        |--------------------------------------------------------------------------
        | Setup
        |--------------------------------------------------------------------------
        */

        $warehouse = Warehouse::where('type', Warehouse::TYPE_AMAZON_FBA)->first();

        Setting::updateOrCreate(['key' => Setting::KEY_WAREHOUSE_IGNORE_AMAZON_FBA_TYPE], [
            'key' => Setting::KEY_WAREHOUSE_IGNORE_AMAZON_FBA_TYPE,
            'value' => false,
        ]);

        Setting::updateOrCreate(['key' => Setting::KEY_WAREHOUSE_PRIORITY], [
            'key' => Setting::KEY_WAREHOUSE_PRIORITY,
            'value' => json_encode([Warehouse::first()->id, $warehouse->id]),
        ]);

        $salesChannelProduct = AmazonProduct::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'json_object' => [
                'seller_sku' => 'ABC',
            ]
        ]);

        $product = Product::factory()->create([
            'sku' => 'ABC',
        ]);

        ProductListing::factory()->create([
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'product_id' => $product->id,
            'listing_sku' => 'ABC',
            'sales_channel_listing_id' => 'ABC',
            'document_id' => $salesChannelProduct->id,
            'document_type' => AmazonProduct::class,
        ]);

        AmazonFbaReportInventory::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'json_object' => [
                'sku' => 'ABC',
                'fnsku' => 'XYZ',
                'afn_fulfillable_quantity' => 10,
            ]
        ]);

        $salesOrder = app(SalesOrderManager::class)->createOrder([
            'sales_order_number' => '123',
            'sales_channel_id' => 1,
            'store_id' => 1,
            'order_status' => SalesOrder::STATUS_OPEN,
            'order_date' => now(),
            'shipping_address_id' => Address::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'description' => 'Test Product',
                    'product_id' => $product->id,
                    'amount' => 10,
                    'quantity' => 1,
                    'product_listing_id' => ProductListing::first()->id,
                ]
            ]
        ]);

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'quantity' => 1,
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Fulfill
        |--------------------------------------------------------------------------
        */

        $amazonCarrier = ShippingCarrier::where('name', 'Amazon MCF')->first();

        $standardShippingMethod = ShippingMethod::query()
            ->where('shipping_carrier_id', $amazonCarrier->id)
            ->where('name', FulfillmentShippingSpeedCategoryEnum::STANDARD->value)
            ->first();

        $response = $this->postJson(route('sales-orders.fulfill', $salesOrder->id), [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_FBA,
            'warehouse_id' => $warehouse->id,
            'metadata' => [
                'shipping_speed' => FulfillmentShippingSpeedCategoryEnum::STANDARD->value,
            ],
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => SalesOrderLine::first()->id,
                    'quantity' => 1,
                    'metadata' => [
                        'fba_seller_sku' => '2291979-FBA-1',
                    ]
                ]
            ]
        ])->assertOk();

        $this->assertDatabaseHas((new AmazonFulfillmentOrder())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'sales_order_fulfillment_id' => SalesOrderFulfillment::first()->id,
            'sellerFulfillmentOrderId' => '123.1',
            'displayableOrderId' => '123',
            'shippingSpeedCategory' => 'Standard',
        ]);

        $this->assertDatabaseHas((new AmazonFulfillmentOrderItem())->getTable(), [
            'amazon_fulfillment_order_id' => AmazonFulfillmentOrder::first()->id,
            'sellerSku' => '2291979-FBA-1',
        ]);

        $this->postJson(route('amazon.outbound.refresh', $this->amazonIntegrationInstance->id))->assertOk();

        $this->assertDatabaseHas((new AmazonFulfillmentOrder())->getTable(), [
            'sellerFulfillmentOrderId' => '123.1',
            'displayableOrderId' => '123',
            'shippingSpeedCategory' => 'Standard',
            'fulfillmentOrderStatus' => 'Complete',
        ]);

        $this->assertDatabaseHas((new AmazonFulfillmentOrderItem())->getTable(), [
            'sellerSku' => '2291979-FBA-1',
            'quantity' => 1,
        ]);

        $this->assertDatabaseHas((new AmazonFulfillmentShipment())->getTable(), [
            'amazon_fulfillment_order_id' => AmazonFulfillmentOrder::first()->id,
            'amazonShipmentId' => "GH6BlBVxF"
        ]);

        $this->assertDatabaseHas((new AmazonFulfillmentShipmentItem())->getTable(), [
            'sellerSku' => '2291979-FBA-1',
            'quantity' => 1,
        ]);

        $this->assertDatabaseHas((new AmazonFulfillmentShipmentPackage())->getTable(), [
            'amazon_fulfillment_shipment_id' => AmazonFulfillmentShipment::first()->id,
            'trackingNumber' => 'TBA313166722760',
        ]);

        $this->assertDatabaseHas((new SalesOrderFulfillment())->getTable(), [
            'sales_order_id' => $salesOrder->id,
            'fulfillment_type' => SalesOrderFulfillment::TYPE_FBA,
            'tracking_number' => 'TBA313166722760',
            'fulfilled_shipping_method' => 'Amazon Logistics',
            'status' => SalesOrderFulfillment::STATUS_FULFILLED,
        ]);
    }
}
