<?php

namespace Modules\Amazon\Tests\Feature\Controllers;

use App\Data\CreateSkuOrderFromSalesChannelData;
use App\Enums\OrderSyncStatusEnum;
use App\Exceptions\ApiException;
use App\Jobs\MapSalesOrderLinesToSalesChannelProductsJob;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Managers\ProductInventoryManager;
use App\Models\Address;
use App\Models\BackorderQueue;
use App\Models\Customer;
use App\Models\FifoLayer;
use App\Models\FinancialLine;
use App\Models\FinancialLineType;
use App\Models\InventoryMovement;
use App\Models\Payment;
use App\Models\PaymentType;
use App\Models\Product;
use App\Models\ProductInventory;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineFinancial;
use App\Models\SalesOrderLineLayer;
use App\Models\User;
use App\Models\Warehouse;
use Exception;
use Illuminate\Foundation\Testing\DatabaseTransactions;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Exceptions\AmazonTimeoutException;
use Modules\Amazon\Jobs\CreateAmazonRefreshOrderItemsJobs;
use Modules\Amazon\Jobs\RefreshAmazonOrderItemsJob;
use Modules\Amazon\Managers\AmazonIntegrationInstanceManager;
use Modules\Amazon\Repositories\AmazonOrderRepository;
use Modules\Amazon\Tests\AmazonMockRequests;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetOrdersAdt;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Entities\AmazonOrderItem;
use Modules\Amazon\Jobs\RefreshAmazonOrdersJob;
use Modules\Amazon\Managers\AmazonOrderManager;
use Modules\Amazon\Tests\AmazonTestingData;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class AmazonOrderControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use AmazonMockRequests;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     * @throws Exception
     */
    public function setUp(): void
    {
        parent::setUp();
        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->has(Warehouse::factory(1)->state([
                'type' => Warehouse::TYPE_AMAZON_FBA,
            ]))
            ->create();

        $integration_settings = $this->amazonIntegrationInstance->integration_settings;
        $integration_settings['proforma_marketplace_cost_percentage'] = 15;
        $this->amazonIntegrationInstance->integration_settings = $integration_settings;
        $this->amazonIntegrationInstance->save();

        PaymentType::factory()->create([
            'name' => 'Amazon Payment',
        ]);

        $this->mockGetAccessToken();
        $this->mockGetRestrictedDataToken();
        $this->mockGetMarketplaceParticipations();

        (new AmazonIntegrationInstanceManager($this->amazonIntegrationInstance))->getMarketplaceParticipations();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_amazon_orders_controller(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Refresh orders
        |--------------------------------------------------------------------------
        */

        $this->mockRefreshOrders();
        $this->mockRefreshOrderItems();

        $this->postJson(route('amazon.orders.refresh', $this->amazonIntegrationInstance->id))->assertOk();

        Queue::assertPushed(RefreshAmazonOrdersJob::class);

        (new AmazonOrderManager($this->amazonIntegrationInstance))->refreshOrders(new AmazonGetOrdersAdt());

        $firstOrder = AmazonTestingData::getOrders()['payload']['Orders'][0];
        $firstOrder['BuyerInfo'] = encryptArray($firstOrder['BuyerInfo']);
        $firstOrder['ShippingAddress'] = encryptArray($firstOrder['ShippingAddress'], [
            'AddressLine1',
            'AddressLine2',
            'AddressLine3',
        ]);

        $this->assertDatabaseHas((new AmazonOrder())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'json_object' => json_encode($firstOrder),
            'error_log' => OrderSyncStatusEnum::PENDING_ITEMS,
        ]);

        Queue::assertPushed(CreateAmazonRefreshOrderItemsJobs::class);
        (new CreateAmazonRefreshOrderItemsJobs($this->amazonIntegrationInstance))->handle();
        Queue::assertPushed(RefreshAmazonOrderItemsJob::class);
        $amazonOrderIds = app(AmazonOrderRepository::class)->getActiveItemlessOrderIds(2);
        (new AmazonOrderManager($this->amazonIntegrationInstance))->refreshOrderItems($amazonOrderIds->toArray());

        $this->assertDatabaseHas((new AmazonOrderItem())->getTable(), [
            'json_object' => json_encode(AmazonTestingData::getOrderItems()['payload']['OrderItems'][0]),
        ]);

        $this->assertDatabaseHas((new AmazonOrder())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'json_object' => json_encode($firstOrder),
            'error_log' => null,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Get orders
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('amazon.orders.index', $this->amazonIntegrationInstance->id), [
            'sortObjs' => [
                "column" => "id",
                "ascending" => true
            ]
        ])->assertOk();

        $response->assertJsonStructure([
            'data' => [
                '*' => [
                    'AmazonOrderId',
                    'sku_sales_order',
                    'PurchaseDateUtc',
                    'LastUpdateDate',
                    'OrderStatus',
                    'error_log',
                    'FulfillmentChannel',
                    'ShipServiceLevel',
                    'OrderCurrency',
                    'OrderTotal',
                    'BuyerName',
                ],
            ],
        ]);

        $firstOrderId = $response->json('data')[0]['id'];

        /*
        |--------------------------------------------------------------------------
        | Show order
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('amazon.orders.show', [$this->amazonIntegrationInstance->id, $firstOrderId]))->assertOk();

        $response->assertJsonStructure([
            'data' => [
                'id',
                'integration_instance_id',
                'sku_sales_order',
                'items',
                'json_object',
                'AmazonOrderId',
                'SellerOrderId',
                'PurchaseDate',
                'PurchaseDateUtc',
                'AmazonPurchaseDate',
                'LastUpdateDate',
                'OrderStatus',
                'error_log',
                'FulfillmentChannel',
                'SalesChannel',
                'ShipServiceLevel',
                'OrderCurrency',
                'OrderTotal',
                'NumberOfItemsShipped',
                'PaymentMethod',
                'MarketplaceId',
                'BuyerName',
                'BuyerEmail',
                'PaymentMethodDetails',
                'ShipmentServiceLevelCategory',
                'OrderType',
                'EarliestShipDate',
                'EarliestShipDateUtc',
                'LatestShipDate',
                'LatestShipDateUtc',
                'EarliestDeliveryDate',
                'EarliestDeliveryDateUtc',
                'LatestDeliveryDate',
                'LatestDeliveryDateUtc',
                'IsBusinessOrder',
                'IsPrime',
                'IsPremiumOrder',
                'ReplacedOrderId',
                'IsReplacementOrder',
                'errors', // TODO: Consider dropping this column
                'archived_at',
                'canceled_at',
                'ShippingAddress',
                'BuyerInfo',
                'created_at',
                'updated_at',
            ],
        ]);

        /*
        |--------------------------------------------------------------------------
        | Create sku orders
        |--------------------------------------------------------------------------
        */

        $salesChannelProduct = AmazonProduct::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'json_object' => [
                'seller_sku' => '1722590-FBA',
            ],
        ]);

        $product = Product::factory()->create([
            'sku' => '1722590-FBA',
            'type' => Product::TYPE_BUNDLE,
        ]);

        $component1 = Product::factory()->create([
            'sku' => 'component1',
        ]);

        $component2 = Product::factory()->create([
            'sku' => 'component2',
        ]);

        $product->setBundleComponents([
            [
                'id' => $component1->id,
                'quantity' => 1,
            ],
            [
                'id' => $component2->id,
                'quantity' => 2,
            ],
        ]);

        ProductListing::factory()->create([
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'product_id' => $product->id,
            'sales_channel_listing_id' => '1722590-FBA',
            'document_id' => $salesChannelProduct->id,
            'document_type' => AmazonProduct::class,
        ]);

        $this->postJson(route('amazon.orders.create-sku-orders', $this->amazonIntegrationInstance->id), CreateSkuOrderFromSalesChannelData::from([
            'create_all_orders' => true,
        ])->toArray())->assertOk();

        $this->assertDatabaseHas((new SalesOrder())->getTable(), [
            'sales_channel_id' => $this->amazonIntegrationInstance->salesChannel->id,
            'sales_order_number' => AmazonTestingData::getOrders()['payload']['Orders'][0]['AmazonOrderId'],
            'sales_channel_order_type' => AmazonOrder::class,
        ]);

        $this->assertDatabaseCount((new SalesOrderLine())->getTable(), 6);
        $this->assertDatabaseCount((new SalesOrderLineFinancial())->getTable(), 6);
        $this->assertDatabaseCount((new Customer())->getTable(), 2);
        $this->assertDatabaseCount((new Address())->getTable(), 4);
        $this->assertDatabaseCount((new Payment())->getTable(), 2);
        // One of the orders is closed due to being FBA so does not have their lines reserved
        $this->assertDatabaseCount((new InventoryMovement())->getTable(), 4);
        $this->assertDatabaseCount((new BackorderQueue())->getTable(), 2);
        // 2 revenue lines for shipping, 2 cost lines for marketplace fee
        $this->assertDatabaseCount((new FinancialLine())->getTable(), 4);
        $this->assertDatabaseCount((new FinancialLineType())->getTable(), 2);
        // For the two mapped lines on the MFN order, each has a total record and warehouse record
        Queue::assertPushed(UpdateProductsInventoryAndAvgCost::class);
        (new ProductInventoryManager([
            $component1->id,
            $component2->id,
        ], true, false))->updateProductInventoryAndAvgCost();
        $this->assertDatabaseCount((new ProductInventory())->getTable(), 4);

        /*
        |--------------------------------------------------------------------------
        | Export orders
        |--------------------------------------------------------------------------
        */

        $this->getJson(route('amazon.orders.export', $this->amazonIntegrationInstance->id))->assertOk();
    }
}
