<?php

namespace Feature\Controllers;

use App\Data\IdSelectionData;
use App\Helpers;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\SalesChannel;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\User;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundPlanAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundShipmentAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundShipmentItemsAdt;
use Modules\Amazon\Entities\AmazonFbaInboundShipFromMapping;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonNewFbaInboundPlan;
use Modules\Amazon\Entities\AmazonNewFbaInboundPlanItem;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipment;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipmentItem;
use Modules\Amazon\Jobs\CreateAmazonRefreshFbaInboundPlanItemsJobs;
use Modules\Amazon\Jobs\CreateAmazonRefreshFbaInboundPlanJobs;
use Modules\Amazon\Jobs\CreateAmazonRefreshFbaInboundShipmentsJobs;
use Modules\Amazon\Jobs\RefreshAmazonNewFbaInboundShipmentItemsJob;
use Modules\Amazon\Managers\AmazonNewInboundManager;
use Modules\Amazon\Repositories\AmazonNewFbaInboundPlanRepository;
use Modules\Amazon\Tests\AmazonMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class AmazonNewFbaInboundShipmentControllerTest extends TestCase
{
    use AmazonMockRequests;
    use FastRefreshDatabase;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();
        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->has(Warehouse::factory(1, ['type' => Warehouse::TYPE_AMAZON_FBA]))
            ->create();

        $this->mockGetAccessToken();
        $this->mockGetRestrictedDataToken();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Throwable
     */
    public function test_amazon_new_inbound_controller(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Refresh inbound plans
        |--------------------------------------------------------------------------
        */

        $this->mockListInboundPlans();
        $this->mockListInboundPlanItems();
        $this->mockGetNewInboundShipment();
        $this->mockListShipmentItems();

        $this->postJson(route('amazon.new-inbound.plans.refresh', $this->amazonIntegrationInstance->id))->assertOk();

        Queue::assertPushed(CreateAmazonRefreshFbaInboundPlanItemsJobs::class);
        Queue::assertPushed(CreateAmazonRefreshFbaInboundPlanJobs::class);
        Queue::assertPushed(CreateAmazonRefreshFbaInboundShipmentsJobs::class);

        $planIds = app(AmazonNewFbaInboundPlanRepository::class)->getActiveItemlessPlanIds();

        $this->assertCount(2, $planIds);

        (new AmazonNewInboundManager($this->amazonIntegrationInstance))->refreshPlanItems([$planIds[0]]);
        (new AmazonNewInboundManager($this->amazonIntegrationInstance))->refreshPlanItems([$planIds[1]]);

        $this->assertDatabaseCount((new AmazonNewFbaInboundPlan())->getTable(), 2);
        $this->assertDatabaseCount((new AmazonNewFbaInboundPlanItem())->getTable(), 4);

        $planIdsNeedingDetail = app(AmazonNewFbaInboundPlanRepository::class)->getPlansNeedingDetail();

        foreach ($planIdsNeedingDetail as $planId) {
            (new AmazonNewInboundManager($this->amazonIntegrationInstance))->getPlan(new AmazonGetInboundPlanAdt(
                planId: $planId
            ));
        }

        $shipments = app(AmazonNewFbaInboundPlanRepository::class)->getShipmentsNeedingDetails($this->amazonIntegrationInstance);

        $firstPlan = $planIds->first();
        $firstShipment = $shipments[$firstPlan][0];

        $secondPlan = $planIds->last();
        $secondShipment = $shipments[$secondPlan][0];

        $warehouseTransfer = WarehouseTransfer::factory()->create([
            'warehouse_transfer_number' => 'FBA17LS62V5D',
            'from_warehouse_id' => Warehouse::first()->id,
            'to_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
        ]);

        // Legacy shipment to be replaced
        AmazonFbaInboundShipment::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'sku_link_id' => $warehouseTransfer->id,
            'sku_link_type' => WarehouseTransfer::class,
            'json_object' =>[
                'ShipmentId' => 'FBA17LS62V5D',
            ]
        ]);

        $this->assertDatabaseHas((new AmazonFbaInboundShipment())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'ShipmentId' => 'FBA17LS62V5D',
        ]);

        (new AmazonNewInboundManager($this->amazonIntegrationInstance))->getShipment(new AmazonGetInboundShipmentAdt(
            planId: $firstPlan,
            shipmentId: $firstShipment
        ));

        $this->assertDatabaseEmpty((new AmazonFbaInboundShipment())->getTable());

        (new AmazonNewInboundManager($this->amazonIntegrationInstance))->getShipment(new AmazonGetInboundShipmentAdt(
            planId: $secondPlan,
            shipmentId: $secondShipment
        ));

        Queue::assertPushed(RefreshAmazonNewFbaInboundShipmentItemsJob::class);

        (new AmazonNewInboundManager($this->amazonIntegrationInstance))->getShipmentItems(new AmazonGetInboundShipmentItemsAdt(
            planId: $firstPlan,
            shipmentId: $firstShipment
        ));

        (new AmazonNewInboundManager($this->amazonIntegrationInstance))->getShipmentItems(new AmazonGetInboundShipmentItemsAdt(
            planId: $secondPlan,
            shipmentId: $secondShipment
        ));

        $this->assertDatabaseHas((new AmazonNewFbaInboundShipment())->getTable(), [
            'shipmentId' => $firstShipment,
            'sku_link_id' => $warehouseTransfer->id,
            'sku_link_type' => WarehouseTransfer::class,
        ]);
        $this->assertDatabaseHas((new AmazonNewFbaInboundShipment())->getTable(), [
            'shipmentId' => $secondShipment
        ]);

        $this->assertDatabaseCount((new AmazonNewFbaInboundShipment())->getTable(), 2);

        $inboundShipment1 = AmazonNewFbaInboundShipment::first();
        $inboundShipment2 = AmazonNewFbaInboundShipment::offset(1)->first();

        $this->assertDatabaseHas((new AmazonNewFbaInboundShipmentItem())->getTable(), [
            'amazon_new_fba_inbound_shipment_id' => $inboundShipment1->id,
        ]);
        $this->assertDatabaseHas((new AmazonNewFbaInboundShipmentItem())->getTable(), [
            'amazon_new_fba_inbound_shipment_id' => $inboundShipment2->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Get inbound plans
        |--------------------------------------------------------------------------
        */

        $this->getJson(route('amazon.new-inbound.plans.index', $this->amazonIntegrationInstance->id))->assertOk();

        /*
        |--------------------------------------------------------------------------
        | Get inbound shipments
        |--------------------------------------------------------------------------
        */

        $this->getJson(route('amazon.new-inbound.shipments.index', $this->amazonIntegrationInstance->id))->assertOk();

        /*
        |--------------------------------------------------------------------------
        | Process inbound shipments
        |--------------------------------------------------------------------------
        */

        $warehouse = Warehouse::factory()->create([
            'name' => 'Warehouse ABC',
        ])->withDefaultLocation();

        $supplier = Supplier::factory()->create([
            'name' => 'AIMS Power',
        ]);

        AmazonFbaInboundShipFromMapping::whereName('Warehouse ABC')->update([
            'link_type' => Warehouse::class,
            'link_id' => $warehouse->id,
        ]);

        AmazonFbaInboundShipFromMapping::whereName('AIMS Power')->update([
            'link_type' => Supplier::class,
            'link_id' => $supplier->id,
        ]);

        $product1 = Product::factory()->create([
            'sku' => 'X003DG2ZIT',
            'name' => 'AIMS Power 2000 Watt 12 Volt Pure Sine Inverter Charger',
        ]);

        $product2 = Product::factory()->create([
            'sku' => 'X003DG2ZIT2',
            'name' => 'AIMS Power 2000 Watt 12 Volt Pure Sine Inverter Charger 2',
        ]);

        $product1->setInitialInventory($warehouse->id, 100, 5);
        $product2->setInitialInventory($warehouse->id, 100, 5);

        SupplierProduct::factory()->create([
            'supplier_id' => $supplier->id,
            'product_id' => $product1->id,
        ]);
        SupplierProduct::factory()->create([
            'supplier_id' => $supplier->id,
            'product_id' => $product2->id,
        ]);

        AmazonFnskuProduct::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fnsku' => $product1->sku,
            'location' => 'US',
            'disposition' => 'SELLABLE',
            'product_id' => $product1->id,
        ]);
        AmazonFnskuProduct::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fnsku' => $product2->sku,
            'location' => 'US',
            'disposition' => 'SELLABLE',
            'product_id' => $product2->id,
        ]);

        $this->postJson(route('amazon.new-inbound.shipments.process', $this->amazonIntegrationInstance->id),
            IdSelectionData::from([
                'ids' => AmazonNewFbaInboundShipment::all()->map(function (AmazonNewFbaInboundShipment $shipment) {
                    return $shipment->id;
                  })->toArray(),
            ])->toArray()
        )->assertOk();

        $this->assertDatabaseHas((new WarehouseTransfer())->getTable(), [
            'warehouse_transfer_number' => 'FBA17LS62V61',
            'from_warehouse_id' => $warehouse->id,
            'to_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
        ]);

        $this->assertDatabaseHas((new WarehouseTransferLine())->getTable(), [
            'product_id' => $product1->id,
        ]);
        $this->assertDatabaseHas((new WarehouseTransferLine())->getTable(), [
            'product_id' => $product2->id,
        ]);

        $this->assertDatabaseHas((new PurchaseOrder())->getTable(), [
            'purchase_order_number' => 'FBA17LS62V5D',
            'supplier_id' => $supplier->id,
            'store_id' => Helpers::setting(Setting::KEY_PO_DEFAULT_STORE),
            'destination_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
        ]);

        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'product_id' => $product1->id,
        ]);
        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'product_id' => $product2->id,
        ]);
    }
}
