<?php

namespace Modules\Amazon\Tests\Feature\Controllers;

use App\Data\IdSelectionData;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\SalesChannel;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\User;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Models\WarehouseTransferLine;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\Amazon\Entities\AmazonFbaInboundShipFromMapping;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonFbaInboundShipmentItem;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipment;
use Modules\Amazon\Jobs\CreateAmazonRefreshFbaInboundShipmentItemsJobs;
use Modules\Amazon\Jobs\RefreshAmazonFbaInboundShipmentItemsJob;
use Modules\Amazon\Managers\AmazonInboundManager;
use Modules\Amazon\Repositories\AmazonFbaInboundShipmentRepository;
use Modules\Amazon\Tests\AmazonMockRequests;
use Modules\Amazon\Tests\AmazonTestingData;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class AmazonFbaInboundShipmentControllerTest extends TestCase
{
    use AmazonMockRequests;
    use FastRefreshDatabase;

    private AmazonIntegrationInstance $amazonIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();
        $this->amazonIntegrationInstance = AmazonIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->has(Warehouse::factory(1, ['type' => Warehouse::TYPE_AMAZON_FBA]))
            ->create();

        $this->mockGetAccessToken();
        $this->mockGetRestrictedDataToken();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Throwable
     */
    public function test_amazon_inbound_controller(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Refresh inbound shipments
        |--------------------------------------------------------------------------
        */

        $this->mockGetInboundShipments();
        $this->mockGetInboundShipmentItems();

        // Create new inbound shipment to make sure the legacy inbound does not duplicate it
        AmazonNewFbaInboundShipment::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'json_object' => [
                'shipmentConfirmationId' => 'FBA17LS62V61'
            ]
        ]);
        $this->postJson(route('amazon.inbound.refresh', $this->amazonIntegrationInstance->id))->assertOk();
        $this->assertDatabaseCount((new AmazonFbaInboundShipment())->getTable(), 1);
        $this->assertDatabaseMissing((new AmazonFbaInboundShipment())->getTable(), [
            'ShipmentId' => 'FBA17LS62V61',
        ]);
        AmazonNewFbaInboundShipment::first()->delete();

        $this->postJson(route('amazon.inbound.refresh', $this->amazonIntegrationInstance->id))->assertOk();

        $this->assertDatabaseHas((new AmazonFbaInboundShipment())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'ShipmentId' => 'FBA17LS62V61',
        ]);
        $this->assertDatabaseCount((new AmazonFbaInboundShipment())->getTable(), 2);

        Queue::assertPushed(CreateAmazonRefreshFbaInboundShipmentItemsJobs::class);
        (new CreateAmazonRefreshFbaInboundShipmentItemsJobs($this->amazonIntegrationInstance))->handle();
        Queue::assertPushed(RefreshAmazonFbaInboundShipmentItemsJob::class);
        $shipmentIds = app(AmazonFbaInboundShipmentRepository::class)->getActiveItemlessShipmentIds()->toArray();
        (new AmazonInboundManager($this->amazonIntegrationInstance))->refreshShipmentItems($shipmentIds);

        $this->assertDatabaseHas((new AmazonFbaInboundShipmentItem())->getTable(), [
            'json_object' => json_encode(AmazonTestingData::getInboundShipmentItems()['payload']['ItemData'][0]),
        ]);

        $this->assertDatabaseHas((new AmazonFbaInboundShipFromMapping())->getTable(), [
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'name' => 'AIMS Power',
        ]);

        /*
        |--------------------------------------------------------------------------
        | Process inbound shipments
        |--------------------------------------------------------------------------
        */

        $warehouse = Warehouse::factory()->create([
            'name' => 'Warehouse ABC',
        ])->withDefaultLocation();

        $supplier = Supplier::factory()->create([
            'name' => 'AIMS Power',
        ]);

        AmazonFbaInboundShipFromMapping::whereName('Warehouse ABC')->update([
            'link_type' => Warehouse::class,
            'link_id' => $warehouse->id,
        ]);

        AmazonFbaInboundShipFromMapping::whereName('AIMS Power')->update([
            'link_type' => Supplier::class,
            'link_id' => $supplier->id,
        ]);

        $product1 = Product::factory()->create([
            'sku' => 'X003DG2ZIT',
            'name' => 'AIMS Power 2000 Watt 12 Volt Pure Sine Inverter Charger',
        ]);

        $product2 = Product::factory()->create([
            'sku' => 'X003DG2ZIT2',
            'name' => 'AIMS Power 2000 Watt 12 Volt Pure Sine Inverter Charger 2',
        ]);

        $product1->setInitialInventory($warehouse->id, 100, 5);

        $product2->setInitialInventory($warehouse->id, 100, 5);

        SupplierProduct::factory()->create([
            'supplier_id' => $supplier->id,
            'product_id' => $product1->id,
        ]);
        SupplierProduct::factory()->create([
            'supplier_id' => $supplier->id,
            'product_id' => $product2->id,
        ]);

        AmazonFnskuProduct::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fnsku' => $product1->sku,
            'location' => 'US',
            'disposition' => 'SELLABLE',
            'product_id' => $product1->id,
        ]);
        AmazonFnskuProduct::factory()->create([
            'integration_instance_id' => $this->amazonIntegrationInstance->id,
            'fnsku' => $product2->sku,
            'location' => 'US',
            'disposition' => 'SELLABLE',
            'product_id' => $product2->id,
        ]);

        $this->postJson(route('amazon.inbound.process', $this->amazonIntegrationInstance->id),
            IdSelectionData::from([
                'ids' => AmazonFbaInboundShipment::all()->map(function (AmazonFbaInboundShipment $shipment) {
                    return $shipment->id;
                  })->toArray(),
            ])->toArray()
        )->assertOk();

        $this->assertDatabaseHas((new WarehouseTransfer())->getTable(), [
            'warehouse_transfer_number' => 'FBA17LS62V5D',
            'from_warehouse_id' => $warehouse->id,
            'to_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
        ]);

        $this->assertDatabaseHas((new WarehouseTransferLine())->getTable(), [
            'product_id' => $product1->id,
        ]);
        $this->assertDatabaseHas((new WarehouseTransferLine())->getTable(), [
            'product_id' => $product2->id,
        ]);

        $this->assertDatabaseHas((new PurchaseOrder())->getTable(), [
            'purchase_order_number' => 'FBA17LS62V61',
            'supplier_id' => $supplier->id,
            'destination_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
        ]);

        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'product_id' => $product1->id,
        ]);
        $this->assertDatabaseHas((new PurchaseOrderLine())->getTable(), [
            'product_id' => $product2->id,
        ]);
    }
}
