<?php

namespace Modules\Amazon\Services;

use App\Enums\HttpMethodEnum;
use App\Exceptions\ApiException;
use Carbon\Carbon;
use Illuminate\Http\Client\ConnectionException;
use Modules\Amazon\Data\AmazonFbaInboundShipmentData;
use Modules\Amazon\Data\AmazonFbaInboundShipmentItemData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Enums\Entities\FbaInboundQueryTypeEnum;
use Modules\Amazon\Enums\Entities\FbaInboundShipmentStatusEnum;
use Modules\Amazon\Exceptions\AmazonTimeoutException;

class AmazonShipmentClient extends AmazonAuthenticationClient
{
    /**
     * TODO: Check pagination.
     *
     */
    public function getFbaInboundShipments(
        ?array $shipmentStatusList = null,
        ?array $shipmentIdList = null,
        ?string $lastUpdatedAfter = null,
        ?string $lastUpdatedBefore = null,
        ?string $nextToken = null,
    ): AmazonResponseData {
        if ($shipmentStatusList || $shipmentIdList) {
            $queryType = FbaInboundQueryTypeEnum::SHIPMENT->value;
        } elseif ($lastUpdatedAfter || $lastUpdatedBefore) {
            $queryType = FbaInboundQueryTypeEnum::DATE_RANGE->value;
        } else {
            $queryType = FbaInboundQueryTypeEnum::NEXT_TOKEN->value;
        }

        /*
         * The documentation says that MarketplaceId is required, but it is not.
         * Documentation: https://developer-docs.amazon.com/sp-api/docs/fulfillment-inbound-api-v0-reference#getshipments
         * Bug report: https://github.com/amzn/selling-partner-api-docs/issues/3133
         */

        $requestParams = [
            'LastUpdatedAfter' => $lastUpdatedAfter ? Carbon::parse($lastUpdatedAfter)->toIso8601ZuluString() : null,
            'LastUpdatedBefore' => $lastUpdatedBefore ?
                Carbon::parse($lastUpdatedBefore)->toIso8601ZuluString() : Carbon::now()->toIso8601ZuluString(),
            'ShipmentStatusList' => implode(',', FbaInboundShipmentStatusEnum::values()),
            'QueryType' => $queryType,
        ];

        if (! empty($shipmentIdList)) {
            $requestParams['ShipmentIdList'] = implode(',', $shipmentIdList);
        }

        if (! is_null($nextToken)) {
            $requestParams['QueryType'] = 'NEXT_TOKEN';
            $requestParams['NextToken'] = $nextToken;
        }

        $response = $this->request(HttpMethodEnum::GET, '/fba/inbound/v0/shipments', $requestParams);

        $collection = AmazonFbaInboundShipmentData::collection($response->json()['payload']['ShipmentData']);

        return AmazonResponseData::from([
            'collection' => $collection,
            'nextToken' => @$response->json()['payload']['NextToken'],
        ]);
    }

    // TODO: Refactor to get inbound items for a single shipment id at a time similar to amazon get order items

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getFbaInboundShipmentItems(string $shipmentId): AmazonResponseData
    {
        /*
         * The documentation says that MarketplaceId is required, but it is not.
         * Documentation: https://developer-docs.amazon.com/sp-api/docs/fulfillment-inbound-api-v0-reference#getshipments
         * Bug report: https://github.com/amzn/selling-partner-api-docs/issues/3133
         */

        $response = $this->request(HttpMethodEnum::GET, '/fba/inbound/v0/shipments/'.$shipmentId.'/items');

        $collection = AmazonFbaInboundShipmentItemData::collection($response->json()['payload']['ItemData']);

        return AmazonResponseData::from([
            'collection' => $collection,
            'nextToken' => @$response->json()['payload']['NextToken'],
        ]);
    }
}
