<?php

namespace Modules\Amazon\Services;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Enums\HttpMethodEnum;
use App\Exceptions\ApiException;
use Carbon\Carbon;
use Exception;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Support\Facades\Http;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetReportsAdt;
use Modules\Amazon\Data\AmazonReportData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Exceptions\AmazonTimeoutException;
use Throwable;

class AmazonReportClient extends AmazonAuthenticationClient
{
    protected string $clientPath;

    public function __construct(IntegrationInstanceInterface $integrationInstance)
    {
        parent::__construct($integrationInstance);
        $this->clientPath = config('amazon.reports_path');
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getReports(AmazonGetReportsAdt $parameters): AmazonResponseData
    {
        $response = $this->request(HttpMethodEnum::GET, $this->clientPath.'reports', $parameters->transform());

        $collection = AmazonReportData::collection($response->json()['reports']);

        return AmazonResponseData::from([
            'collection' => $collection,
            'nextToken' => @$response->json()['payload']['NextToken'],
        ]);
    }

    /**
     * @throws Throwable
     * @throws ConnectionException
     * @throws AmazonTimeoutException
     * @throws ApiException
     */
    public function createReport(AmazonReportTypeEnum $reportType, ?Carbon $start = null, ?Carbon $end = null, ?array $options = null, ?array $marketplace_ids = null): int
    {
        $request = [
            'reportType' => $reportType(),
            'dataStartTime' => $start->toIso8601ZuluString(),
            'dataEndTime' => $end?->toIso8601ZuluString(),
            'reportOptions' => $options,
            'marketplaceIds' => $marketplace_ids ?? $this->integrationInstance->getMarketplaceIds()
        ];

        $response = $this->request(HttpMethodEnum::POST, '/reports/2021-06-30/reports', $request)->json();

        throw_if(! isset($response['reportId']), "Report $reportType->value failed to create.  Please check API log for more details.");

        return $response['reportId'];
    }

    /**
     * Get report details - https://developer-docs.amazon.com/sp-api/docs/reports-api-v2021-06-30-reference#getreport
     *
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getReportById(string $reportId): AmazonReportData
    {
        return AmazonReportData::from($this->request(HttpMethodEnum::GET, "/reports/2021-06-30/reports/$reportId")->json());
    }

    /**
     * Download the report as content.
     */
    public function downloadReportAsContent(string $url): string
    {
        return Http::get($url)->body();
    }

    /**
     * Get the information required for retrieving a report document's contents and download the report.
     *
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getReportDocument(string $reportDocumentId): string
    {
        $reportDocumentDetails = $this->request(HttpMethodEnum::GET, "/reports/2021-06-30/documents/$reportDocumentId")->json();

        $content = $this->downloadReportAsContent($reportDocumentDetails['url']);

        if (isset($reportDocumentDetails['compressionAlgorithm'])) {
            $content = match ($reportDocumentDetails['compressionAlgorithm']) {
                'GZIP' => gzdecode($content),
                default => throw new Exception('Unknown compression type on '.$reportDocumentId),
            };
        }

        return $content;
    }
}
