<?php

namespace Modules\Amazon\Services;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Enums\HttpMethodEnum;
use App\Exceptions\ApiException;
use Illuminate\Http\Client\ConnectionException;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetOrdersAdt;
use Modules\Amazon\Data\AmazonOrderData;
use Modules\Amazon\Data\AmazonOrderItemData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Exceptions\AmazonTemporaryBadRequestException;
use Modules\Amazon\Exceptions\AmazonTimeoutException;
use Modules\Amazon\Repositories\AmazonOrderRepository;

class AmazonOrderClient extends AmazonAuthenticationClient
{
    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     * @throws AmazonTemporaryBadRequestException
     */
    public function getOrders(AmazonGetOrdersAdt|ApiDataTransformerInterface $parameters): AmazonResponseData
    {
        $requestParams = $parameters->transform();

        // TODO: Not sure where the best place to have this logic is
        if (!@$requestParams['CreatedAfter'] && !@$requestParams['LastUpdatedAfter']) {
            $requestParams['CreatedAfter'] = app(AmazonOrderRepository::class)->getStartDateForNew($this->integrationInstance)->toIso8601ZuluString();
        }

        $this->getRestrictedDataToken(HttpMethodEnum::GET, '/orders/v0/orders', ['buyerInfo', 'shippingAddress']);

        $this->isRestrictedResource = true;

        // We can't build this in the Adt because integration instance is not available there
        $requestParams['MarketplaceIds'] = $this->integrationInstance->getMarketplaceIdsCsv();

        $response = $this->request(HttpMethodEnum::GET, '/orders/v0/orders', $requestParams);

        if ($response->status() == 400) {
            throw new AmazonTemporaryBadRequestException($response->json()['errors'][0]['message']);
        }

        $collection = AmazonOrderData::collection($response->json()['payload']['Orders']);

        return AmazonResponseData::from([
            'collection' => $collection->toCollection(),
            'nextToken' => @$response->json()['payload']['NextToken'],
        ]);
    }

    /**
     * @throws ApiException
     * @throws ConnectionException
     * @throws AmazonTimeoutException
     */
    public function getOrder(string $orderId): AmazonResponseData
    {
        $this->getRestrictedDataToken(HttpMethodEnum::GET, "/orders/v0/orders/$orderId", ['buyerInfo', 'shippingAddress']);

        $this->isRestrictedResource = true;

        $response = $this->request(HttpMethodEnum::GET, "/orders/v0/orders/$orderId");

        $collection = AmazonOrderData::collection([$response->json()['payload']]);

        return AmazonResponseData::from([
            'collection' => $collection,
            'nextToken' => null,
        ]);
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getOrderItems(string $orderId): AmazonResponseData
    {
        $this->isRestrictedResource = false;

        $response = $this->request(HttpMethodEnum::GET, '/orders/v0/orders/'.$orderId.'/orderItems');

        $collection = AmazonOrderItemData::collection($response->json()['payload']['OrderItems']);

        return AmazonResponseData::from([
            'collection' => $collection,
            'nextToken' => @$response->json()['payload']['NextToken'],
        ]);
    }
}
