<?php

namespace Modules\Amazon\Services;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Enums\HttpMethodEnum;
use App\Exceptions\ApiException;
use Illuminate\Http\Client\ConnectionException;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundPlanAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundPlansAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundShipmentAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundShipmentItemsAdt;
use Modules\Amazon\Data\AmazonFbaInboundPlanData;
use Modules\Amazon\Data\AmazonFbaInboundPlanItemData;
use Modules\Amazon\Data\AmazonNewFbaInboundShipmentItemData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Exceptions\AmazonTimeoutException;

class AmazonNewInboundClient extends AmazonAuthenticationClient
{
    public function __construct(IntegrationInstanceInterface $integrationInstance)
    {
        parent::__construct($integrationInstance);
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function listInboundPlans(AmazonGetInboundPlansAdt $parameters): AmazonResponseData
    {
        $requestParams = $parameters->transform();

        $response = $this->request(HttpMethodEnum::GET, '/inbound/fba/2024-03-20/inboundPlans', $requestParams);

        $collection = AmazonFbaInboundPlanData::collection($response->json()['inboundPlans']);

        return AmazonResponseData::from([
            'collection' => $collection,
            'nextToken' => @$response->json()['pagination']['nextToken'],
        ]);
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getInboundPlan(AmazonGetInboundPlanAdt $parameters): array
    {
        $response = $this->request(HttpMethodEnum::GET, '/inbound/fba/2024-03-20/inboundPlans/' . $parameters->planId);

        if ($response->status() === 400) {
            return [];
        }

        return $response->json();
    }

    // TODO: Refactor to get inbound items for a single shipment id at a time similar to amazon get order items

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getFbaInboundPlanItems(string $planId): AmazonResponseData
    {
        $response = $this->request(HttpMethodEnum::GET, '/inbound/fba/2024-03-20/inboundPlans/'.$planId.'/items');

        $collection = AmazonFbaInboundPlanItemData::collection($response->json()['items']);

        return AmazonResponseData::from([
            'collection' => $collection,
            'nextToken' => @$response->json()['pagination']['nextToken'],
        ]);
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getFbaInboundShipment(AmazonGetInboundShipmentAdt $parameters): array
    {
        $response = $this->request(HttpMethodEnum::GET, '/inbound/fba/2024-03-20/inboundPlans/' . $parameters->planId . '/shipments/' . $parameters->shipmentId);

        return $response->json();
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function listShipmentItems(AmazonGetInboundShipmentItemsAdt $parameters): AmazonResponseData
    {
        $response = $this->request(HttpMethodEnum::GET, '/inbound/fba/2024-03-20/inboundPlans/' . $parameters->planId . '/shipments/' . $parameters->shipmentId . '/items');

        $collection = AmazonNewFbaInboundShipmentItemData::collection($response->json()['items']);

        return AmazonResponseData::from([
            'collection' => $collection,
            'nextToken' => @$response->json()['pagination']['nextToken'],
        ]);
    }
}
