<?php

namespace Modules\Amazon\Services;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Enums\HttpMethodEnum;
use App\Exceptions\ApiException;
use App\Helpers\ArrayToXML;
use App\Models\SalesChannel;
use DOMException;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\RequestOptions;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Support\Collection;
use Modules\Amazon\ApiDataTransferObjects\AmazonFulfillOrdersAdt;
use Modules\Amazon\Data\AmazonCreateFeedData;
use Modules\Amazon\Data\AmazonCreateOrderFulfillmentFeedData;
use Modules\Amazon\Data\AmazonFeedDocumentData;
use Modules\Amazon\Enums\Entities\AmazonFeedTypeEnum;
use Modules\Amazon\Exceptions\AmazonFeedException;
use Modules\Amazon\Exceptions\AmazonTimeoutException;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Throwable;

class AmazonFeedClient extends AmazonAuthenticationClient
{
    /**
     * @throws GuzzleException
     */
    public function uploadFeedDocument(AmazonFeedDocumentData $feedDocument, string $feedData): int
    {
        customlog('amazon', 'upload feed document', [
            'feedData' => $feedData,
        ]);

        return (new \GuzzleHttp\Client())
            ->put($feedDocument->url, [
                RequestOptions::HEADERS => [
                    'content-type' => 'text/xml;charset=UTF-8',
                    'host' => parse_url($feedDocument->url, PHP_URL_HOST),
                ],
                RequestOptions::BODY => $feedData,
            ])
            ->getStatusCode();
    }

    /**
     * @throws Exception
     */
    private function createFeedDocument(): AmazonFeedDocumentData
    {
        $response = $this->request(HttpMethodEnum::POST, '/feeds/2021-06-30/documents', [
            'contentType' => 'text/xml;charset=UTF-8',
        ]);

        return AmazonFeedDocumentData::from($response->json());
    }

    /**
     * @throws DOMException
     */
    private function wrapXmlFeed($feedInputArray): string
    {
        return (new ArrayToXML($feedInputArray, [
            'rootElementName' => 'AmazonEnvelope',
            '_attributes' => [
                'xmlns:xsi' => 'http://www.w3.org/2001/XMLSchema-instance',
                'xsi:noNamespaceSchemaLocation' => 'amzn-envelope.xsd',
            ],
        ], true, 'UTF-8'))->prettify()->toXml();
    }

    /**
     * Create POST_ORDER_ACKNOWLEDGEMENT_DATA feed.
     *
     * @throws ApiException
     * @throws GuzzleException
     * @throws Throwable
     * @throws DOMException
     */
    public function createAcknowledgeOrderFeed(#[DataCollectionOf(AmazonCreateOrderAcknowledgementFeedDto::class)] $amazonOrders): AmazonCreateFeedData
    {
        if (app()->environment() !== 'production') {
            throw new Exception('POST_ORDER_ACKNOWLEDGEMENT_DATA - Production environment required.', 1);
        }

        //If no records, stop.
        throw_if($amazonOrders->count() === 0);

        //Messages
        $messages = $amazonOrders->map(function ($amazonOrder, $index) {
            return [
                'MessageID' => $index + 1,
                'OrderAcknowledgement' => [
                    'AmazonOrderID' => $amazonOrder->AmazonOrderId,
                    'MerchantOrderID' => $amazonOrder->MerchantOrderID,
                    'StatusCode' => $amazonOrder->StatusCode,
                    'Item' => [
                        'AmazonOrderItemCode' => $amazonOrder->Item_AmazonOrderItemCode,
                        'MerchantOrderItemID' => $amazonOrder->Item_MerchantOrderItemID,
                    ],
                ],
            ];
        });

        //Make XML
        $feedXml = $this->wrapXmlFeed([
            'Header' => [
                'DocumentVersion' => '1.01',
                'MerchantIdentifier' => $this->merchantId,
            ],
            'MessageType' => 'OrderAcknowledgement',
            'Message' => $messages->toArray(),
        ]);

        //Create Feed Document
        $feedDocument = $this->createFeedDocument();

        //Upload feed
        if ($this->uploadFeedDocument($feedDocument, $feedXml) === 200) {
            $feedResponse = $this->request(HttpMethodEnum::POST, '/feeds/2021-06-30/feeds', [
                'feedType' => AmazonFeedTypeEnum::ORDER_ACKNOWLEDGEMENT(),
                'marketplaceIds' => $this->integrationInstance->getMarketplaceIdsCsv(),
                'inputFeedDocumentId' => $feedDocument->feedDocumentId,
            ]);

            if ($feedResponse->status() > 200 and $feedResponse->status() < 300) {
                return AmazonCreateFeedData::from([
                    'feedId' => $feedResponse->json()['feedId'],
                    'inputFeedDocumentId' => $feedDocument->feedDocumentId,
                ]);
            } else {
                throw new Exception('Unable to upload feed', 1);
            }
        } else {
            throw new Exception('Unable to upload feed', 1);
        }
    }

    /**
     * @throws AmazonFeedException
     * @throws Throwable
     * @throws ConnectionException
     * @throws AmazonTimeoutException
     * @throws DOMException
     * @throws GuzzleException
     * @throws ApiException
     */
    public function fulfillOrders(AmazonFulfillOrdersAdt|ApiDataTransformerInterface $parameters): AmazonCreateFeedData
    {
        $requestParams = $parameters->transform();
        throw_if($parameters->messages->count() === 0);

        $requestParams['Header']['MerchantIdentifier'] = $this->merchantId;

        $feedXml = $this->wrapXmlFeed($requestParams);

        if (app()->environment() !== 'production') {
            return AmazonCreateFeedData::from([
                'feedId' => 'testing-'.fake()->sha1,
                'inputFeedDocumentId' => 'testing-'.fake()->sha1,
                'requestFeedDocumentXml' => $feedXml,
            ]);
        }

        //Create Feed Document
        $feedDocument = $this->createFeedDocument();

        //Upload feed
        if ($this->uploadFeedDocument($feedDocument, $feedXml) === 200) {
            $feedResponse = $this->request(HttpMethodEnum::POST, '/feeds/2021-06-30/feeds', [
                'feedType' => AmazonFeedTypeEnum::ORDER_FULFILLMENT(),
                'marketplaceIds' => $this->integrationInstance->getMarketplaceIds(),
                'inputFeedDocumentId' => $feedDocument->feedDocumentId,
            ]);

            if ($feedResponse->status() > 200 and $feedResponse->status() < 300) {
                /*
                 * TODO: @Jatin currently we don't use the response here.  I think it is likely we would need to manage the feeds.
                 *  For Amazon we don't want to retry a submission if a feed already was submitted so we do want to update the submitted_to_sales_channel_at
                 *  which is already done in the caller of fulfillOrders... but what if the feed fails?  We would want to clear any submitted_to_sales_channel_at at
                 *  that point to allow it to retry.  This needs to be managed.  We have an AmazonFeedRequest and AmazonFeedSubmission model that we could use to manage this.
                 *  Currently it is not being in this process.  We should use it.  Not sure the difference between the two models.
                 */
                return AmazonCreateFeedData::from([
                    'feedId' => $feedResponse->json()['feedId'],
                    'inputFeedDocumentId' => $feedDocument->feedDocumentId,
                    'requestFeedDocumentXml' => $feedXml,
                ]);
            } else {
                throw new AmazonFeedException('Unable to upload order fulfillment feed', 1);
            }
        } else {
            throw new Exception('Unable to upload feed', 1);
        }
    }

    /**
     * Create POST_ORDER_ACKNOWLEDGEMENT_DATA feed.
     *
     * @deprecated replaced by fulfillOrders
     *
     * @throws ApiException
     * @throws GuzzleException
     * @throws Throwable
     * @throws DOMException
     */
    public function createOrderFulfillmentFeed(#[DataCollectionOf(AmazonCreateOrderFulfillmentFeedData::class)] $amazonOrders): AmazonCreateFeedData
    {
        if (app()->environment() !== 'production') {
            throw new Exception('POST_ORDER_ACKNOWLEDGEMENT_DATA - Production environment required.', 1);
        }

        //If no records, stop.
        throw_if($amazonOrders->count() === 0);
        $messages = collect([]);

        //Messages
        $amazonOrders->map(function ($amazonOrder, $index) use (&$messages) {
            $messages->push([
                'MessageID' => $index + 1,
                'OrderFulfillment' => [
                    'AmazonOrderID' => $amazonOrder->AmazonOrderID,
                    'MerchantFulfillmentID' => $amazonOrder->MerchantFulfillmentID,
                    'FulfillmentDate' => $amazonOrder->FulfillmentDate,
                    'FulfillmentData' => [
                        'CarrierCode' => $amazonOrder->FulfillmentData_CarrierName,
                        'ShippingMethod' => SalesChannel::UNSPECIFIED_SHIPPING_METHOD,
                        'ShipperTrackingNumber' => $amazonOrder->FulfillmentData_ShipperTrackingNumber,
                    ],
                    'Item' => [
                        'AmazonOrderItemCode' => $amazonOrder->Item->AmazonOrderItemCode,
                        'Quantity' => $amazonOrder->Item->Quantity,
                        'MerchantFulfillmentItemID' => $amazonOrder->Item->MerchantFulfillmentItemID,
                    ],
                ],
            ]);
        });

        //Make XML
        $feedXml = $this->wrapXmlFeed([
            'Header' => [
                'DocumentVersion' => '1.01',
                'MerchantIdentifier' => $this->merchantId,
            ],
            'MessageType' => 'OrderFulfillment',
            'Message' => $messages->toArray(),
        ]);

        //Create Feed Document
        $feedDocument = $this->createFeedDocument();

        //Upload feed
        if ($this->uploadFeedDocument($feedDocument, $feedXml) === 200) {
            $feedResponse = $this->request(HttpMethodEnum::POST, '/feeds/2021-06-30/feeds', [
                'feedType' => AmazonFeedTypeEnum::ORDER_FULFILLMENT(),
                'marketplaceIds' => $this->integrationInstance->getMarketplaceIdsCsv(),
                'inputFeedDocumentId' => $feedDocument->feedDocumentId,
            ]);

            if ($feedResponse->status() > 200 and $feedResponse->status() < 300) {
                return AmazonCreateFeedData::from([
                    'feedId' => $feedResponse->json()['feedId'],
                    'inputFeedDocumentId' => $feedDocument->feedDocumentId,
                ]);
            } else {
                throw new Exception('Unable to upload feed', 1);
            }
        } else {
            throw new Exception('Unable to upload feed', 1);
        }
    }

    /**
     * Create POST_PRODUCT_PRICING_DATA feed.
     *
     * @throws ApiException
     * @throws GuzzleException
     * @throws Throwable
     * @throws DOMException
     */
    public function createPricingFeed(Collection $productListings): AmazonCreateFeedData
    {
        if (app()->environment() !== 'production') {
            throw new Exception('POST_PRODUCT_PRICING_DATA - Production environment required.', 1);
        }

        //If no records
        throw_if($productListings->count() === 0);

        //Messages
        $messages = $productListings->map(function ($productListing, $index) {
            return [
                'MessageID' => $index + 1,
                'Price' => [
                    'SKU' => $productListing->SKU,
                    // 'StandardPrice' => $productListing->StandardPrice,
                    'StandardPrice' => [
                        '_attributes' => ['currency' => 'USD'],
                        '_value' => "$productListing->StandardPrice",
                    ],

                ],
            ];
        });

        //Make XML
        $feedXml = $this->wrapXmlFeed([
            'Header' => [
                'DocumentVersion' => '1.01',
                'MerchantIdentifier' => $this->merchantId,
            ],
            'MessageType' => 'Price',
            'Message' => $messages->toArray(),
        ]);

        //Create Feed Document
        $feedDocument = $this->createFeedDocument();

        //Upload feed
        if ($this->uploadFeedDocument($feedDocument, $feedXml) === 200) {
            $feedResponse = $this->request(HttpMethodEnum::POST, '/feeds/2021-06-30/feeds', [
                'feedType' => AmazonFeedTypeEnum::PRICING(),
                'marketplaceIds' => $this->integrationInstance->getMarketplaceIdsCsv(),
                'inputFeedDocumentId' => $feedDocument->feedDocumentId,
            ]);

            if ($feedResponse->status() > 200 and $feedResponse->status() < 300) {
                return AmazonCreateFeedData::from([
                    'feedId' => $feedResponse->json()['feedId'],
                    'inputFeedDocumentId' => $feedDocument->feedDocumentId,
                ]);
            } else {
                throw new Exception('Unable to upload feed', 1);
            }
        } else {
            throw new Exception('Unable to upload feed', 1);
        }
    }

    /**
     * Create POST_INVENTORY_AVAILABILITY_DATA feed.
     *
     * @throws ApiException
     * @throws GuzzleException
     * @throws Throwable
     * @throws DOMException
     */
    public function createInventoryFeed(Collection $productListings): AmazonCreateFeedData
    {
        if (app()->environment() !== 'production') {
            throw new Exception('POST_INVENTORY_AVAILABILITY_DATA - Production environment required.', 1);
        }

        //If no records
        throw_if($productListings->count() === 0);

        //Messages
        $messages = $productListings->map(function ($productListing, $index) {
            return [
                'MessageID' => $index + 1,
                'OperationType' => 'Update',
                'Inventory' => $productListing->toArray(),
            ];
        });

        //Make XML
        $feedXml = $this->wrapXmlFeed([
            'Header' => [
                'DocumentVersion' => '1.01',
                'MerchantIdentifier' => $this->merchantId,
            ],
            'MessageType' => 'Inventory',
            'Message' => $messages->toArray(),
        ]);

        //Create Feed Document
        $feedDocument = $this->createFeedDocument();

        //Upload feed
        if ($this->uploadFeedDocument($feedDocument, $feedXml) === 200) {
            $feedResponse = $this->request(HttpMethodEnum::POST, '/feeds/2021-06-30/feeds', [
                'feedType' => AmazonFeedTypeEnum::INVENTORY(),
                'marketplaceIds' => $this->integrationInstance->getMarketplaceIdsCsv(),
                'inputFeedDocumentId' => $feedDocument->feedDocumentId,
            ]);

            if ($feedResponse->status() > 200 and $feedResponse->status() < 300) {
                return AmazonCreateFeedData::from([
                    'feedId' => $feedResponse->json()['feedId'],
                    'inputFeedDocumentId' => $feedDocument->feedDocumentId,
                ]);
            } else {
                throw new Exception('Unable to upload feed', 1);
            }
        } else {
            throw new Exception('Unable to upload feed', 1);
        }
    }
}
