<?php

namespace Modules\Amazon\Services;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\ClientResponseDataInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\SalesChannelApiHasSeparateOrderItemsEndpointInterface;
use App\Abstractions\Integrations\SalesChannels\SalesChannelClientInterface;
use App\Enums\HttpMethodEnum;
use App\Exceptions\ApiException;
use Carbon\Carbon;
use DOMException;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundPlanAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundPlansAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundShipmentAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetInboundShipmentItemsAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetFinancialEventsAdt;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetReportsAdt;
use Modules\Amazon\Data\AmazonCatalogItemData;
use Modules\Amazon\Data\AmazonCreateFeedData;
use Modules\Amazon\Data\AmazonCreateOrderAcknowledgementFeedData;
use Modules\Amazon\Data\AmazonCreateOrderFulfillmentFeedData;
use Modules\Amazon\Data\AmazonFeedData;
use Modules\Amazon\Data\AmazonFulfillmentOrderData;
use Modules\Amazon\Data\AmazonMarketplaceParticipationsData;
use Modules\Amazon\Data\AmazonReportData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Entities\AmazonFinancialEventGroup;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Exceptions\AmazonTimeoutException;
use Spatie\LaravelData\Attributes\DataCollectionOf;
use Spatie\LaravelData\DataCollection;
use Throwable;

/**
 * For readability, we are separating the Authentication and main client calls into separate classes
 *
 * For Amazon, there are many different API sections there are each pretty detailed, so for readability, we separate
 * them into one class per section.
 */
class AmazonClient extends AmazonAuthenticationClient implements SalesChannelApiHasSeparateOrderItemsEndpointInterface, SalesChannelClientInterface
{
    private AmazonReportClient $reportClient;

    private AmazonOrderClient $orderClient;

    private AmazonProductClient $productClient;

    private AmazonFinancialClient $financialClient;

    private AmazonShipmentClient $shipmentClient;

    private AmazonNewInboundClient $newInboundClient;

    private AmazonFeedClient $feedClient;

    private AmazonFulfillmentClient $fulfillmentClient;

    public function __construct(IntegrationInstanceInterface $integrationInstance)
    {
        parent::__construct($integrationInstance);

        $this->reportClient = new AmazonReportClient($integrationInstance);
        $this->orderClient = new AmazonOrderClient($integrationInstance);
        $this->productClient = new AmazonProductClient($integrationInstance);
        $this->financialClient = new AmazonFinancialClient($integrationInstance);
        $this->shipmentClient = new AmazonShipmentClient($integrationInstance);
        $this->newInboundClient = new AmazonNewInboundClient($integrationInstance);
        $this->feedClient = new AmazonFeedClient($integrationInstance);
        $this->fulfillmentClient = new AmazonFulfillmentClient($integrationInstance);
    }

    /*
    |--------------------------------------------------------------------------
    | Reporting
    |--------------------------------------------------------------------------
    */

    /**
     * TODO: This is an example of request parameter abstraction possibilities... need to implement throughout
     * TODO: This is an example of delegating to a specific client for a specific type of api call... need to implement throughout
     *
     * @throws ApiException
     */
    public function getReports(AmazonGetReportsAdt $parameters): AmazonResponseData
    {
        return $this->reportClient->getReports($parameters);
    }

    public function createReport(AmazonReportTypeEnum $reportType, ?Carbon $start = null, ?Carbon $end = null, ?array $options = null, ?array $marketplace_ids = null): int
    {
        return $this->reportClient->createReport($reportType, $start, $end, $options, $marketplace_ids);
    }

    /**
     * @throws ApiException
     */
    public function getReportById(string $reportId): AmazonReportData
    {
        return $this->reportClient->getReportById($reportId);
    }

    /**
     * @throws GuzzleException
     */
    public function downloadReportAsContent(string $url): void
    {
        $this->reportClient->downloadReportAsContent($url);
    }

    /**
     * @throws GuzzleException
     */
    public function getReportDocument(string $reportDocumentId): string
    {
        return $this->reportClient->getReportDocument($reportDocumentId);
    }

    /*
    |--------------------------------------------------------------------------
    | Orders
    |--------------------------------------------------------------------------
    */
    public function getOrders(ApiDataTransformerInterface $parameters): AmazonResponseData
    {
        return $this->orderClient->getOrders($parameters);
    }

    /**
     * @throws Exception
     */
    public function getOrder(string $orderId): ClientResponseDataInterface
    {
        return $this->orderClient->getOrder($orderId);
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getOrderItems(string $orderId): AmazonResponseData
    {
        return $this->orderClient->getOrderItems($orderId);
    }

    /*
    |--------------------------------------------------------------------------
    | Products
    |--------------------------------------------------------------------------
    */

    /**
     * Get information about a catalog item.
     *
     * @throws Exception
     */
    public function getCatalogItem(string $asin): ?AmazonCatalogItemData
    {
        return $this->productClient->getCatalogItem($asin);
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function searchCatalogItems(array $asins): DataCollection
    {
        return $this->productClient->searchCatalogItems($asins);
    }

    public function getProduct(string $productId): array
    {
        // Amazon products are downloaded via reports
        return [];
    }

    public function getProducts(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        // Amazon products are downloaded via reports
        return AmazonResponseData::from([
            'collection' => collect(),
            'nextToken' => collect(),
        ]);
    }

    /*
    |--------------------------------------------------------------------------
    | Finance API
    |--------------------------------------------------------------------------
    */

    /**
     * Returns financial event groups for a given date range.
     *
     * @throws ApiException
     */
    public function getFinancialEventGroups(
        ?string $startedAfter = null,
        ?string $startedBefore = null,
        ?string $nextToken = null
    ): ClientResponseDataInterface {
        return $this->financialClient->getFinancialEventGroups($startedAfter, $startedBefore, $nextToken);
    }

    /**
     * Returns financial events by group id for a given date range.
     *
     * @throws Exception
     */
    public function getFinancialEventsByGroupId(
        AmazonFinancialEventGroup $amazonFinancialEventGroup,
        ?string $postedAfter = null,
        ?string $nextToken = null
    ): ClientResponseDataInterface {
        return $this->financialClient->getFinancialEventsByGroupId($amazonFinancialEventGroup, $postedAfter, $nextToken);
    }

    /**
     * Returns financial events for a given date range.
     *
     * @throws Exception
     */
    public function getFinancialEvents(AmazonGetFinancialEventsAdt $parameters): ClientResponseDataInterface
    {
        return $this->financialClient->getFinancialEvents($parameters);
    }

    /*
    |--------------------------------------------------------------------------
    | FBA Inbound Shipments
    |--------------------------------------------------------------------------
    */

    /**
     * TODO: Check pagination.
     *
     * @throws Exception
     */
    public function getFbaInboundShipments(
        ?array $shipmentStatusList = null,
        ?array $shipmentIdList = null,
        ?string $lastUpdatedAfter = null,
        ?string $lastUpdatedBefore = null,
        ?string $nextToken = null,
    ): ClientResponseDataInterface {
        return $this->shipmentClient->getFbaInboundShipments(
            $shipmentStatusList,
            $shipmentIdList,
            $lastUpdatedAfter,
            $lastUpdatedBefore,
            $nextToken
        );
    }

    /*
    |--------------------------------------------------------------------------
    | New FBA Inbound Shipments
    |--------------------------------------------------------------------------
    */

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function listInboundPlans(AmazonGetInboundPlansAdt $parameters): ClientResponseDataInterface {
        return $this->newInboundClient->listInboundPlans($parameters);
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getInboundPlan(AmazonGetInboundPlanAdt $parameters): array {
        return $this->newInboundClient->getInboundPlan($parameters);
    }

    /**
     * @throws Exception
     */
    public function getFbaInboundShipmentItems(string $shipmentId): ClientResponseDataInterface
    {
        return $this->shipmentClient->getFbaInboundShipmentItems($shipmentId);
    }

    /*
    |--------------------------------------------------------------------------
    | Feeds
    |--------------------------------------------------------------------------
    */

    /**
     * Get the data for feed document.
     *
     * @throws ApiException
     */
    public function getResultFeedDocument(string $resultFeedDocumentId): string
    {
        // dd(self::download($this->request(HttpMethodEnum::GET, '/feeds/2021-06-30/documents/'.$resultFeedDocumentId, [])->json()['url']));
        return $this->request(HttpMethodEnum::GET, '/feeds/2021-06-30/documents/'.$resultFeedDocumentId, [])->json()['url'];
    }

    /**
     * Get feed details.
     */
    public function getFeed(string $feedId): AmazonFeedData
    {
        return AmazonFeedData::from($this->request(HttpMethodEnum::GET, '/feeds/2021-06-30/feeds/'.$feedId)->json());
    }

    public function getResultFeedDocumentXml(string $url): string
    {
        $result = Http::get($url);

        if ($result->ok()) {
            return $result->body();
        }
    }

    /**
     * Create POST_INVENTORY_AVAILABILITY_DATA feed.
     *
     * @throws ApiException
     * @throws GuzzleException
     * @throws Throwable
     * @throws DOMException
     */
    public function createInventoryFeed(Collection $productListings): AmazonCreateFeedData
    {
        return $this->feedClient->createInventoryFeed($productListings);
    }

    /**
     * Create POST_PRODUCT_PRICING_DATA feed.
     *
     * @throws ApiException
     * @throws GuzzleException
     * @throws Throwable
     * @throws DOMException
     */
    public function createPricingFeed(Collection $productListings): AmazonCreateFeedData
    {
        return $this->feedClient->createPricingFeed($productListings);
    }

    /**
     * Create POST_ORDER_ACKNOWLEDGEMENT_DATA feed.
     *
     * @throws ApiException
     * @throws GuzzleException
     * @throws Throwable
     * @throws DOMException
     */
    public function createAcknowledgeOrderFeed(#[DataCollectionOf(AmazonCreateOrderAcknowledgementFeedData::class)] $amazonOrders): AmazonCreateFeedData
    {
        return $this->feedClient->createAcknowledgeOrderFeed($amazonOrders);
    }

    /**
     * Create POST_ORDER_ACKNOWLEDGEMENT_DATA feed.
     *
     * @throws ApiException
     * @throws GuzzleException
     * @throws Throwable
     * @throws DOMException
     */
    public function createOrderFulfillmentFeed(#[DataCollectionOf(AmazonCreateOrderFulfillmentFeedData::class)] $amazonOrders): AmazonCreateFeedData
    {
        return $this->feedClient->createAcknowledgeOrderFeed($amazonOrders);
    }

    /*
    |--------------------------------------------------------------------------
    | Fulfillment Outbound
    |--------------------------------------------------------------------------
    */

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function createFulfillmentOrder(AmazonFulfillmentOrderData $dto): void
    {
        $this->fulfillmentClient->createFulfillmentOrder($dto);
    }

    public function getFulfillmentOrder(string $sellerFulfillmentOrderId): AmazonResponseData
    {
        return $this->fulfillmentClient->getFulfillmentOrder($sellerFulfillmentOrderId);
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function listAllFulfillmentOrders(Carbon $queryStartDate, ?string $nextToken = null): AmazonResponseData
    {
        return $this->fulfillmentClient->listAllFulfillmentOrders($queryStartDate, $nextToken);
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function cancelFulfillmentOrder(string $sellerFulfillmentOrderId): void
    {
        $this->fulfillmentClient->cancelFulfillmentOrder($sellerFulfillmentOrderId);
    }

    public function fulfillOrder(ApiDataTransformerInterface $parameters): void
    {
        // Not implemented for Amazon
    }

    public function fulfillOrders(ApiDataTransformerInterface $parameters): AmazonCreateFeedData
    {
        return $this->feedClient->fulfillOrders($parameters);
    }

    public function syncInventory(ApiDataTransformerInterface $parameters)
    {
        // TODO: Implement syncInventory() method.
    }

    public function getFbaInboundPlanItems(string $planId): AmazonResponseData
    {
        return $this->newInboundClient->getFbaInboundPlanItems($planId);
    }

    public function getFbaInboundShipment (AmazonGetInboundShipmentAdt $parameters)
    {
        return $this->newInboundClient->getFbaInboundShipment($parameters);
    }

    public function listShipmentItems(AmazonGetInboundShipmentItemsAdt $parameters)
    {
        return $this->newInboundClient->listShipmentItems($parameters);
    }

    /*
   |--------------------------------------------------------------------------
   | Seller
   |--------------------------------------------------------------------------
   */

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getMarketplaceParticipations(): AmazonResponseData
    {
        $response = $this->request(HttpMethodEnum::GET, '/sellers/v1/marketplaceParticipations');

        $collection = AmazonMarketplaceParticipationsData::collection($response->json()['payload']);

        return AmazonResponseData::from([
            'collection' => $collection,
        ]);
    }
}
