<?php

namespace Modules\Amazon\Services;

use App\Abstractions\Integrations\AbstractIntegrationClient;
use App\Abstractions\Integrations\Concerns\OauthAuthenticationTrait;
use App\Abstractions\Integrations\IntegrationClientInterface;
use App\Abstractions\Integrations\IntegrationHeaderDtoInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\OauthAuthenticationClientInterface;
use App\Enums\HttpMethodEnum;
use App\Exceptions\ApiException;
use App\Exceptions\OauthRefreshTokenFailureException;
use App\Models\ApiLog;
use DateTime;
use Exception;
use GuzzleHttp\Exception\ConnectException;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Http\Client\PendingRequest;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use Modules\Amazon\Data\AmazonHeaderData;
use Modules\Amazon\Data\AmazonRefreshTokenResponseData;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Exceptions\AmazonCouldNotResolveHostException;
use Modules\Amazon\Exceptions\AmazonTimeoutException;
use Modules\Amazon\Managers\AmazonRegionManager;

class AmazonAuthenticationClient extends AbstractIntegrationClient implements IntegrationClientInterface, OauthAuthenticationClientInterface
{
    use OauthAuthenticationTrait;

    private string $awsClient;

    private string $awsSecret;

    private string $lwaClient;

    private string $lwaSecret;

    private string $awsRegion;

    private string $path;

    private string $httpMethod;

    private array $parameters;

    // Amazon access token generated from refresh token.
    private ?string $accessToken = null;

    private ?string $restrictedAccessToken = null;

    protected bool $isRestrictedResource = false;

    protected ?string $merchantId;

    /**
     * @throws Exception
     */
    public function __construct(IntegrationInstanceInterface $integrationInstance)
    {
        parent::__construct($integrationInstance);
        $this->awsClient = config('amazon.aws_client_id', '');
        $this->awsSecret = config('amazon.aws_client_secret', '');
        //$this->lwaClientAppId = ''; //TODO
        $this->lwaClient = config('amazon.lwa_client_id', '');
        $this->lwaSecret = config('amazon.lwa_client_secret', '');

        if ($integrationInstance->refresh_token) {
            $this->setClientCredentials($integrationInstance->refresh_token);
        }

        if ($integrationInstance->country) {
            $this->setRegion($integrationInstance);
        }

        $this->merchantId = @$this->integrationInstance['connection_settings']['selling_partner_id'];
    }

    public function client(): PendingRequest
    {
        return Http::withHeaders($this->requestHeaders);
    }

    public function getRedirectUrl(): string
    {
        $params = [
            'state' => config('app.url').'_'.$this->integrationInstance->id,
            'application_id' => config('amazon.lwa_app_id'),
            'redirect_uri' => config('amazon.lwa_callback_url'),
        ];

        if (config('amazon.lwa_beta')) {
            $params['version'] = 'beta';
        }

        $amazonIntegrationInstance = AmazonIntegrationInstance::find($this->integrationInstance->id);

        return (new AmazonRegionManager($amazonIntegrationInstance))->getSellerCentralUrl().config('amazon.url_authorize').'?'.http_build_query($params);
    }

    public function getRefreshToken(): string
    {
        return $this->refreshToken;
    }

    /**
     * @throws Exception
     */
    public function setRegion(IntegrationInstanceInterface $integrationInstance): self
    {
        $regionConfig = config('amazon.regions')[$integrationInstance->country];
        $this->setBaseUrl($regionConfig['baseUrl']);
        $this->awsRegion = $regionConfig['awsRegion'];

        return $this;
    }

    /**
     * Get the refresh token from Amazon.
     *
     * @throws OauthRefreshTokenFailureException
     */
    public function getRefreshTokenFromAuthCode(string $authCode): AmazonRefreshTokenResponseData
    {
        $response = Http::post(config('amazon.url_access_token'), [
            'grant_type' => 'authorization_code',
            'code' => $authCode,
            'redirect_uri' => config('amazon.lwa_callback_url'),
            'client_id' => config('amazon.lwa_client_id'),
            'client_secret' => config('amazon.lwa_client_secret'),
        ]);

        if (! $response->ok()) {
            throw new OauthRefreshTokenFailureException($response->json()['error_description']);
        }

        return AmazonRefreshTokenResponseData::from($response->json());
    }

    /**
     * Generate access_token using refresh_token.
     *
     * @throws Exception
     */
    public function getAccessToken(): ?string
    {
        if ($this->isRestrictedResource) {
            if (is_null($this->restrictedAccessToken)) {
                throw new Exception('Please set restricted token.', 1);
            }

            return $this->restrictedAccessToken;
        }

        if (is_null($this->accessToken)) {
            try {
                $response = Http::post(
                    config('amazon.url_access_token'),
                    [
                        'grant_type' => 'refresh_token',
                        'client_id' => $this->lwaClient,
                        'client_secret' => $this->lwaSecret,
                        'refresh_token' => $this->getRefreshToken(),
                    ]
                );
            } catch (ConnectionException $e) {
                // Check if the error message contains 'cURL error 6'
                if (str_contains($e->getMessage(), 'cURL error 6')) {
                    throw new AmazonCouldNotResolveHostException('Amazon could not resolve host for '.config('amazon.url_access_token'));
                }
                throw $e;
            } catch (ConnectException $e) {
                throw $e;
            }

            if ($response->ok()) {
                $this->accessToken = $response->json()['access_token'];
            } else {
                throw new Exception('Invalid amazon refresh token.', 1);
            }
        }

        return $this->accessToken;
    }

    /**
     * @throws ApiException
     */
    public function getRestrictedDataToken(HttpMethodEnum $httpMethod, string $path, array $dataElements = []): ?string
    {
        if (is_null($this->restrictedAccessToken)) {
            $response = $this->request(HttpMethodEnum::POST, '/tokens/2021-03-01/restrictedDataToken', [
                'restrictedResources' => [
                    [
                        'method' => strtoupper($httpMethod->value),
                        'path' => $path,
                        'dataElements' => $dataElements,
                    ],
                ],
            ]);

            if ($response->ok()) {
                $this->restrictedAccessToken = $response->json()['restrictedDataToken'];
            } else {
                throw new Exception('Invalid amazon refresh token.', 1);
            }
        }

        return $this->restrictedAccessToken;
    }

    /**
     * @throws Exception
     */
    public function setRequestHeaders(IntegrationHeaderDtoInterface|AmazonHeaderData $headerDto): void
    {
        static $service = 'execute-api';
        static $terminationString = 'aws4_request';
        static $algorithm = 'AWS4-HMAC-SHA256';
        static $phpAlgorithm = 'sha256';
        static $signedHeaders = 'content-type;host;x-amz-access-token;x-amz-date';
        static $contentType = 'application/json';

        $canonicalURI = $this->path;
        $canonicalQueryString = '';
        $httpRequestMethod = $headerDto->httpMethod;
        $parameters = $headerDto->parameters;

        if (strtolower($httpRequestMethod) === strtolower('GET') && ! is_null($parameters)) {
            $canonicalQueryString = http_build_query($parameters);
            $parameters = '';
        } elseif (strtolower($httpRequestMethod) === strtolower('POST')) {
            $parameters = json_encode($parameters);
        } elseif (strtolower($httpRequestMethod) === strtolower('PUT')) {
            $parameters = json_encode($parameters);
        }

        $currentDateTime = new DateTime('UTC');
        $reqDate = $currentDateTime->format('Ymd');
        $reqDateTime = $currentDateTime->format('Ymd\THis\Z');

        // Create signing key
        $kDate = hash_hmac($phpAlgorithm, $reqDate, 'AWS4'.$this->awsSecret, true);
        $kRegion = hash_hmac($phpAlgorithm, $this->awsRegion, $kDate, true);
        $kService = hash_hmac($phpAlgorithm, 'execute-api', $kRegion, true);
        $kSigning = hash_hmac($phpAlgorithm, $terminationString, $kService, true);

        // Create canonical headers
        $canonicalHeaders = [];
        $canonicalHeaders[] = 'content-type:'.$contentType;
        $canonicalHeaders[] = 'host:'.$this->host;
        $canonicalHeaders[] = 'x-amz-access-token:'.$this->getAccessToken();
        $canonicalHeaders[] = 'x-amz-date:'.$reqDateTime;
        $canonicalHeadersStr = implode("\n", $canonicalHeaders);

        // Create request payload
        $requestHashedPayload = hash($phpAlgorithm, $parameters);

        // Create canonical request
        $canonicalRequest = [];
        $canonicalRequest[] = $httpRequestMethod;
        $canonicalRequest[] = $canonicalURI;
        $canonicalRequest[] = $canonicalQueryString;
        $canonicalRequest[] = $canonicalHeadersStr."\n";
        $canonicalRequest[] = $signedHeaders;
        $canonicalRequest[] = $requestHashedPayload;
        $requestCanonicalRequest = implode("\n", $canonicalRequest);
        $requestHashedCanonicalRequest = hash($phpAlgorithm, mb_convert_encoding($requestCanonicalRequest, 'UTF-8'));

        // Create scope
        $credentialScope = [];
        $credentialScope[] = $reqDate;
        $credentialScope[] = $this->awsRegion;
        $credentialScope[] = $service;
        $credentialScope[] = $terminationString;
        $credentialScopeStr = implode('/', $credentialScope);

        // Create string to signing
        $stringToSign = [];
        $stringToSign[] = $algorithm;
        $stringToSign[] = $reqDateTime;
        $stringToSign[] = $credentialScopeStr;
        $stringToSign[] = $requestHashedCanonicalRequest;
        $stringToSignStr = implode("\n", $stringToSign);

        // Create signature
        $signature = hash_hmac($phpAlgorithm, $stringToSignStr, $kSigning);

        // Create authorization header
        $authorizationHeader = [];
        $authorizationHeader[] = 'Credential='.$this->awsClient.'/'.$credentialScopeStr;
        $authorizationHeader[] = 'SignedHeaders='.$signedHeaders;
        $authorizationHeader[] = 'Signature='.($signature);
        $authorizationHeaderStr = $algorithm.' '.implode(', ', $authorizationHeader);

        // Request headers
        $this->requestHeaders = [
            'authorization' => $authorizationHeaderStr,
            'content-length' => strlen($parameters),
            'content-type' => $contentType,
            'host' => $this->host,
            'x-amz-access-token' => $this->getAccessToken(),
            'x-amz-date' => $reqDateTime,
        ];
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function request(HttpMethodEnum $httpMethod, string $path, array $parameters = []): Response
    {
        $this->path = $path;
        $this->httpMethod = strtoupper($httpMethod->value);

        if (count($parameters) > 0) {
            $parameters = collect($parameters)->sortKeys()->toArray();
        }

        $this->parameters = $parameters;

        $this->setRequestHeaders(AmazonHeaderData::from([
            'httpMethod' => $this->httpMethod,
            'parameters' => $this->parameters,
        ]));

        $apiLog = null;
        if ($this->isApiLoggingEnabled) {
            $apiLog = ApiLog::create([
                'integration_instance_id' => $this->integrationInstance->id,
                'url' => $this->getBaseUrl().$path,
                'requestHeaders' => $this->getRequestHeaders(),
                'requestBody' => $this->parameters,
                'responseStatusCode' => null,
                'responseHeaders' => null,
                'responseBody' => null,
            ]);
        }

        try {
            $response = $this->client()->{$httpMethod->value}($this->getBaseUrl().$path, $this->parameters);
        } catch (ConnectionException $e) {
            // Check if the error message contains 'cURL error 28'
            if (str_contains($e->getMessage(), 'cURL error 28')) {
                throw new AmazonTimeoutException('Amazon API timed out. Please try again later.');
            }
            throw $e;
        } catch (ConnectException $e) {
            throw $e;
        }

        if ($this->isApiLoggingEnabled) {
            $apiLog->update([
                'responseStatusCode' => $response->status(),
                'responseHeaders' => $response->headers(),
                'responseBody' => $response->body(),
            ]);
        }

        if ($response->status() === 429) {
            $statusCode = $response->status();
            throw new ApiException('Throttling error', $statusCode);
        }

        return $response;
    }
}
