<?php

namespace Modules\Amazon\Repositories;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelOrderRepository;
use App\Enums\OrderSyncStatusEnum;
use App\Models\PaymentType;
use App\Repositories\PaymentRepository;
use Illuminate\Support\Collection;
use Illuminate\Support\LazyCollection;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Entities\AmazonOrderItem;
use Modules\Amazon\Enums\Entities\AmazonFulfillmentChannelEnum;
use Modules\Amazon\Enums\Entities\OrderStatusEnum;

class AmazonOrderRepository extends AbstractSalesChannelOrderRepository
{
    public function bulkDelete(array $ids): void
    {
        parent::bulkDelete($ids);

        // Delete payment type if no AmazonIntegrationInstance exists
        if (app(AmazonIntegrationInstanceRepository::class)->getTotalCount() === 0) {
            app(PaymentRepository::class)->deletePaymentTypeByName(PaymentType::PAYMENT_TYPE_AMAZON);
        }

        /*
         * TODO: Delete all other amazon entities
         *
         * Check if already taken care of elsewhere first
         *
         * Shipping carriers
         * Shipping methods
         * Shipping integration methods
         * Warehouses
         * etc.
         */
    }

    public function getActiveItemlessOrderIds(int $limit = 30): Collection
    {
        return AmazonOrder::query()
            ->whereIn('OrderStatus', OrderStatusEnum::STATUSES_ACTIVE)
            ->where(function ($query) {
                $query->whereDoesntHave('orderItems')
                    ->orWhere('error_log', OrderSyncStatusEnum::PENDING_ITEMS());
            })
            ->limit($limit)
            ->orderBy('LastUpdateDate')
            ->pluck('AmazonOrderId');
    }

    public function getEarliestPendingOrderLastUpdateDate(): ?string
    {
        return AmazonOrder::query()->where('OrderStatus', OrderStatusEnum::STATUS_PENDING)->oldest('LastUpdateDate')->pluck('LastUpdateDate')->first();
    }

    /**
     * Returns the fulfilled orders where BuyerInfo and ShippingAddress exists.
     */
    public function getFulfilledOrdersForPiiReset(): LazyCollection
    {
        return AmazonOrder::query()
            ->whereHas('salesOrder', function ($query) {
                $query->whereHas('salesOrderFulfillments', function ($query) {
                    $query->whereNotNull('fulfilled_at')
                        ->where('fulfilled_at', '<=', now()->subDays(30));
                });
            })
            ->where(function ($query) {
                $query->orWhere(function ($query) {
                    $query->whereNotNull('ShippingAddress')
                        ->where('ShippingAddress', '!=', '[]');
                })
                    ->orWhereNotNull('BuyerEmail')
                    ->orWhereNotNull('BuyerName');
            })
            ->with([
                'salesOrder' => function ($query) {
                    $query->with([
                        'billingAddress',
                        'shippingAddress',
                        'customer',
                    ]);
                },
            ])
            ->cursor();
    }

    /**
     * Set the BuyerInfo and ShippingAddress to empty array.
     */
    public function resetPiiDataForFulfilledOrders(): void
    {
        $this->getFulfilledOrdersForPiiReset()
            ->each(function ($order) {
                if ($billingAddress = $order->salesOrder?->billingAddress) {
                    $billingAddress->resetPii();
                }

                if ($shippingAddress = $order->salesOrder?->shippingAddress) {
                    $shippingAddress->resetPii();
                }

                if ($customer = $order->salesOrder?->customer) {
                    $customer->resetPii();
                }

                $order->json_object = array_merge($order->json_object, [
                    'BuyerInfo' => [],
                    'ShippingAddress' => [],
                ]);

                $order->update();
            });
    }

    /**
     * @return Collection<array-key, \Illuminate\Database\Eloquent\Builder>
     */
    public function getSalesOrdersForShipping(AmazonIntegrationInstance $amazonIntegrationInstance, string $shipLevelService): Collection
    {
        return AmazonOrder::with([])->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereNotNull('sku_sales_order_id')
            ->where('ShipServiceLevel', $shipLevelService)
            ->get();
    }

    /**
     * @return Collection<array-key, \Illuminate\Database\Eloquent\Builder>
     */
    public function getCanceledOrders(AmazonIntegrationInstance|IntegrationInstanceInterface $integrationInstance, ?Collection $collection = null): Collection
    {
        return parent::getCanceledOrdersQuery($integrationInstance, $collection)
            ->where('OrderStatus', OrderStatusEnum::STATUS_CANCELED())
            ->where('FulfillmentChannel', AmazonFulfillmentChannelEnum::MFN)
            ->get();
    }

    protected static function getModelClassName(): string
    {
        return AmazonOrder::class;
    }

    protected static function getLinesModelClassName(): string
    {
        return AmazonOrderItem::class;
    }

    public function getAmazonOrdersForFinancialLines(AmazonIntegrationInstance|IntegrationInstanceInterface $integrationInstance): Collection
    {
        return AmazonOrder::with([
            'amazonFinancialShipmentEvent',
            'orderItems',
            'salesOrder.salesOrderLines',
        ])
            ->whereHas('orderItems', function ($query) {
                $query->doesntHave('financialLines');
            })
            ->has('salesOrder')
            ->where('integration_instance_id', $integrationInstance->id)
            ->get();
    }
}
