<?php

namespace Modules\Amazon\Repositories;

use App\Models\Product;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Data\AmazonFnskuGenerationData;
use Modules\Amazon\Data\LinkAmazonFnskuProductToSkuProductData;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipmentItem;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReport;
use Spatie\LaravelData\DataCollection;

class AmazonFnskuRepository
{
    public function saveAmazonFnskuProductFromGeneratedData(AmazonIntegrationInstance $integrationInstance, AmazonFnskuGenerationData $fnskuGenerationData, ?int $productId, ?array $potentialProductMatches = null): AmazonFnskuProduct
    {
        $fnskuProduct = AmazonFnskuProduct::query()->updateOrCreate([
            'integration_instance_id' => $integrationInstance->id,
            'fnsku' => $fnskuGenerationData->fnsku,
            'location' => $fnskuGenerationData->location,
            'disposition' => $fnskuGenerationData->disposition,
        ], [
            'integration_instance_id' => $integrationInstance->id,
            'fnsku' => $fnskuGenerationData->fnsku,
            'location' => $fnskuGenerationData->location,
            'disposition' => $fnskuGenerationData->disposition,
            'product_id' => $productId,
            'potential_product_matches' => $potentialProductMatches,
        ]);

        return $fnskuProduct;
    }

    public function fnskusWithoutProducts(AmazonIntegrationInstance $integrationInstance): EloquentCollection
    {
        return AmazonFnskuProduct::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereNull('product_id')
            ->get();
    }

    public function sanitizeFnskuProductsForLedgerReconciliation(AmazonIntegrationInstance $integrationInstance, ?AmazonReport $amazonReport = null, ?array $ids = null): EloquentCollection
    {
        return AmazonFnskuProduct::with('amazonFbaInitialInventory.fifoLayer')
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereHas('amazonFbaReportInventoryLedgers', function ($query) use ($amazonReport, $ids) {
                $query->whereNull('sku_link_id');
                if ($amazonReport) {
                    $query->where('amazon_report_id', $amazonReport->id);
                }
                if ($ids) {
                    $query->whereIn('id', $ids);
                }
            })
            ->whereHas('product')
            ->where(function ($query) {
                $query->whereHas('amazonFbaInitialInventory.fifoLayer'); // Has been reconciled
                $query->orWhereDoesntHave('amazonFbaInitialInventory'); // Doesn't need reconciling because no initial inventory
            })
            ->get();
    }

    public function sanitizeFnskuProductsForLedgerUnreconciliation(AmazonIntegrationInstance $integrationInstance, ?array $ids = null): Collection
    {
        return AmazonFnskuProduct::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->whereHas('amazonFbaReportInventoryLedgers', function ($query) use ($ids) {
                $query->whereNotNull('sku_link_id');
                if ($ids) {
                    $query->whereIn('id', $ids);
                }
            })
            ->get();
    }

    public function getFnskusNeedingGeneration(AmazonIntegrationInstance $integrationInstance, ?AmazonReport $amazonReport = null): DataCollection
    {
        $querySummaries = AmazonFbaInitialInventory::query()
            ->select('amazon_fba_initial_inventory.integration_instance_id', 'amazon_fba_initial_inventory.asin', 'amazon_fba_initial_inventory.fnsku', 'amazon_fba_initial_inventory.location', 'amazon_fba_initial_inventory.disposition')
            ->leftJoin('amazon_fnsku_products', function($join) use ($integrationInstance) {
                $join->on('amazon_fnsku_products.integration_instance_id', '=', 'amazon_fba_initial_inventory.integration_instance_id')
                    ->on('amazon_fnsku_products.fnsku', '=', 'amazon_fba_initial_inventory.fnsku')
                    ->on('amazon_fnsku_products.location', '=', 'amazon_fba_initial_inventory.location')
                    ->on('amazon_fnsku_products.disposition', '=', 'amazon_fba_initial_inventory.disposition');
            })
            ->where('amazon_fba_initial_inventory.integration_instance_id', $integrationInstance->id)
            ->whereNull('amazon_fnsku_products.fnsku')
            ->distinct();

        $queryDetail = AmazonFbaReportInventoryLedger::query()
            ->select('amazon_fba_report_inventory_ledger.integration_instance_id', 'amazon_fba_report_inventory_ledger.asin', 'amazon_fba_report_inventory_ledger.fnsku', DB::raw('amazon_fba_report_inventory_ledger.country as location'), 'amazon_fba_report_inventory_ledger.disposition')
            ->leftJoin('amazon_fnsku_products', function($join) use ($integrationInstance) {
                $join->on('amazon_fnsku_products.integration_instance_id', '=', 'amazon_fba_report_inventory_ledger.integration_instance_id')
                    ->on('amazon_fnsku_products.fnsku', '=', 'amazon_fba_report_inventory_ledger.fnsku')
                    ->on('amazon_fnsku_products.location', '=', DB::raw('amazon_fba_report_inventory_ledger.country'))
                    ->on('amazon_fnsku_products.disposition', '=', 'amazon_fba_report_inventory_ledger.disposition');
            })
            ->where('amazon_fba_report_inventory_ledger.integration_instance_id', $integrationInstance->id)
            ->whereNull('amazon_fnsku_products.fnsku')
            ->distinct();

        if ($amazonReport) {
            $queryDetail->where('amazon_fba_report_inventory_ledger.amazon_report_id', $amazonReport->id);
        }

        $queryInboundShipmentLines = AmazonNewFbaInboundShipmentItem::query()
            ->select('amazon_new_fba_inbound_shipments.integration_instance_id', 'amazon_new_fba_inbound_shipment_items.asin', 'amazon_new_fba_inbound_shipment_items.fnsku', DB::raw('amazon_new_fba_inbound_shipments.destinationCountry as location'), DB::raw('"SELLABLE" as disposition'))
            ->join('amazon_new_fba_inbound_shipments', 'amazon_new_fba_inbound_shipments.id', '=', 'amazon_new_fba_inbound_shipment_items.amazon_new_fba_inbound_shipment_id')
            ->where('amazon_new_fba_inbound_shipments.integration_instance_id', $integrationInstance->id)
            ->leftJoin('amazon_fnsku_products', function($join) use ($integrationInstance) {
                $join->on('amazon_fnsku_products.integration_instance_id', '=', 'amazon_new_fba_inbound_shipments.integration_instance_id')
                    ->on('amazon_fnsku_products.fnsku', '=', 'amazon_new_fba_inbound_shipment_items.fnsku')
                    ->on('amazon_fnsku_products.location', '=', DB::raw('amazon_new_fba_inbound_shipments.destinationCountry'))
                    ->on('amazon_fnsku_products.disposition', '=', DB::raw('"SELLABLE"'));
            })
            ->whereNull('amazon_fnsku_products.fnsku')
            ->distinct();

        return AmazonFnskuGenerationData::collection($querySummaries->union($queryDetail)->union($queryInboundShipmentLines)->distinct()->get());
    }

    public function getFnskuExtendedData(AmazonIntegrationInstance $integrationInstance, Collection $sellerSkus): DataCollection
    {
        $querySummaries = AmazonFnskuProduct::query()
            ->select('amazon_fba_initial_inventory.integration_instance_id', 'amazon_fnsku_products.id', 'amazon_fba_initial_inventory.msku', 'amazon_fba_initial_inventory.asin', 'amazon_fba_initial_inventory.fnsku', 'amazon_fba_initial_inventory.location', 'amazon_fba_initial_inventory.disposition')
            ->join('amazon_fba_initial_inventory', function($join) use ($integrationInstance) {
                $join->on('amazon_fba_initial_inventory.integration_instance_id', '=', 'amazon_fnsku_products.integration_instance_id')
                    ->on('amazon_fba_initial_inventory.fnsku', '=', 'amazon_fnsku_products.fnsku')
                    ->on('amazon_fba_initial_inventory.location', '=', 'amazon_fnsku_products.location')
                    ->on('amazon_fba_initial_inventory.disposition', '=', 'amazon_fnsku_products.disposition');
            })
            ->whereIn('amazon_fba_initial_inventory.msku', $sellerSkus)
            ->where('amazon_fnsku_products.integration_instance_id', $integrationInstance->id);

        $queryDetail = AmazonFnskuProduct::query()
            ->select('amazon_fba_report_inventory_ledger.integration_instance_id', 'amazon_fnsku_products.id', 'amazon_fba_report_inventory_ledger.msku', 'amazon_fba_report_inventory_ledger.asin', 'amazon_fba_report_inventory_ledger.fnsku', DB::raw('amazon_fba_report_inventory_ledger.country as location'), 'amazon_fba_report_inventory_ledger.disposition')
            ->join('amazon_fba_report_inventory_ledger', function($join) use ($integrationInstance) {
                $join->on('amazon_fba_report_inventory_ledger.integration_instance_id', '=', 'amazon_fnsku_products.integration_instance_id')
                    ->on('amazon_fba_report_inventory_ledger.fnsku', '=', 'amazon_fnsku_products.fnsku')
                    ->on('amazon_fba_report_inventory_ledger.country', '=', DB::raw('amazon_fnsku_products.location'))
                    ->on('amazon_fba_report_inventory_ledger.disposition', '=', 'amazon_fnsku_products.disposition');
            })
            ->whereIn('amazon_fba_report_inventory_ledger.msku', $sellerSkus)
            ->where('amazon_fnsku_products.integration_instance_id', $integrationInstance->id);

        $queryInboundShipmentLines = AmazonNewFbaInboundShipmentItem::query()
            ->select('amazon_new_fba_inbound_shipments.integration_instance_id', 'amazon_fnsku_products.id', 'amazon_new_fba_inbound_shipment_items.msku', 'amazon_new_fba_inbound_shipment_items.asin', 'amazon_new_fba_inbound_shipment_items.fnsku', DB::raw('amazon_new_fba_inbound_shipments.destinationCountry as location'), DB::raw('"SELLABLE" as disposition'))
            ->join('amazon_new_fba_inbound_shipments', 'amazon_new_fba_inbound_shipments.id', '=', 'amazon_new_fba_inbound_shipment_items.amazon_new_fba_inbound_shipment_id')
            ->where('amazon_new_fba_inbound_shipments.integration_instance_id', $integrationInstance->id)
            ->leftJoin('amazon_fnsku_products', function($join) use ($integrationInstance) {
                $join->on('amazon_fnsku_products.integration_instance_id', '=', 'amazon_new_fba_inbound_shipments.integration_instance_id')
                    ->on('amazon_fnsku_products.fnsku', '=', 'amazon_new_fba_inbound_shipment_items.fnsku')
                    ->on('amazon_fnsku_products.location', '=', DB::raw('amazon_new_fba_inbound_shipments.destinationCountry'))
                    ->on('amazon_fnsku_products.disposition', '=', DB::raw('"SELLABLE"'));
            })
            ->whereIn('amazon_new_fba_inbound_shipment_items.msku', $sellerSkus)
            ->distinct();

        return AmazonFnskuGenerationData::collection($querySummaries->union($queryDetail)->union($queryInboundShipmentLines)->distinct()->get());
    }

    public function getProductFromFnsku(AmazonIntegrationInstance $integrationInstance, string $fnsku): ?Product
    {
        /** @var AmazonFnskuProduct $amazonFnSkuProduct */
        $amazonFnSkuProduct = AmazonFnskuProduct::query()
            ->where('integration_instance_id', $integrationInstance->id)
            ->where('fnsku', $fnsku)
            ->first();

        return $amazonFnSkuProduct?->product;
    }

    public function getAmazonFnskuProductsFromProduct(Product $product): EloquentCollection
    {
        return AmazonFnskuProduct::query()
            ->where('product_id', $product->id)
            ->get();
    }

    public function getPotentialFnskusForMerchantSku(AmazonIntegrationInstance $amazonIntegrationInstance, string $sku): array
    {
        $fnskus = [];
        // Get all potential fnskus for a given sku
        /** @var AmazonProduct $amazonProduct */
        if ($amazonProduct = AmazonProduct::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('seller_sku', $sku)
            ->first()) {
            $product = $amazonProduct->productListing->product;
            $fnskus = $this->getAmazonFnskuProductsFromProduct($product)->pluck('fnsku')->toArray();
        }

        return $fnskus;
    }

    public function linkFnskuProductToSkuProduct(LinkAmazonFnskuProductToSkuProductData $data): AmazonFnskuProduct
    {
        $amazonFnskuProduct = AmazonFnskuProduct::query()
            ->where('id', $data->id)
            ->firstOrFail();
        $amazonFnskuProduct->product_id = $data->product_id;
        $amazonFnskuProduct->potential_product_matches = null;
        $amazonFnskuProduct->save();
        return $amazonFnskuProduct;
    }

    /**
     * For when trying to reconcile a ledger when there could be unreconciled ledgers before it
     */
    public function countUnreconciledLedgersBeforeLedger(AmazonFbaReportInventoryLedger $ledger): int
    {
        $unreconciledLedgersQuery = $ledger->amazonFnskuProduct->amazonFbaReportInventoryLedgers()
            ->where('integration_instance_id', $ledger->integration_instance_id)
            ->whereNull('reconciled_at')
            ->where('event_datetime', '<', $ledger->event_datetime);

        return $unreconciledLedgersQuery->count();
    }

    /**
     * For when trying to unreconcile a ledger when there could be reconciled ledgers after it
     */
    public function countReconciledLedgersAfterLedger(AmazonFbaReportInventoryLedger $ledger): int
    {
        $reconciledLedgersQuery = $ledger->amazonFnskuProduct->amazonFbaReportInventoryLedgers()
            ->where('integration_instance_id', $ledger->integration_instance_id)
            ->whereNotNull('reconciled_at')
            ->where('event_datetime', '>', $ledger->event_datetime);

        return $reconciledLedgersQuery->count();
    }
}