<?php

namespace Modules\Amazon\Repositories;

use App\Models\AccountingTransaction;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Modules\Amazon\Data\AmazonFinancialEventGroupData;
use Modules\Amazon\Entities\AmazonFinancialEventGroup;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonReportSettlementData;
use Modules\Amazon\Entities\AmazonReportSettlementTypeMapping;
use Modules\Amazon\Enums\Entities\AmazonFinancialEventGroupAccountingStatusEnum;
use Modules\Amazon\Enums\Entities\AmazonFinancialEventGroupProcessingStatusEnum;
use Modules\Amazon\Jobs\CreateAmazonFinancialEventGroupAccountingTransactionsJob;

class AmazonFinancialEventGroupRepository
{
    public function save(AmazonIntegrationInstance $amazonIntegrationInstance, Collection $groups): void
    {
        /** @var AmazonFinancialEventGroupData $group */
        foreach ($groups as $group) {
            if ($amazonFinancialEventGroupIds = $this->getFromFinancialEventGroupId($amazonIntegrationInstance->id, $group->json_object['FinancialEventGroupId'])) {
                $amazonFinancialEventGroupIds->update([
                    'FinancialEventGroupStart_datetime' => Carbon::parse($group->json_object['FinancialEventGroupStart']),
                    'FinancialEventGroupEnd_datetime' => Carbon::parse(@$group->json_object['FinancialEventGroupEnd']),
                    'json_object' => $group->json_object,
                ]);
            } else {
                AmazonFinancialEventGroup::query()->create(array_merge(
                    ['integration_instance_id' => $amazonIntegrationInstance->id],
                    [
                        'FinancialEventGroupStart_datetime' => Carbon::parse($group->json_object['FinancialEventGroupStart']),
                        'FinancialEventGroupEnd_datetime' => Carbon::parse(@$group->json_object['FinancialEventGroupEnd']),
                        'json_object' => $group->json_object,
                    ]
                ));
            }
        }
        dispatch(new CreateAmazonFinancialEventGroupAccountingTransactionsJob($amazonIntegrationInstance));
    }

    public function getFromFinancialEventGroupId(int $amazonIntegrationInstanceId, string $FinancialEventGroupId): ?AmazonFinancialEventGroup
    {
        /** @var AmazonFinancialEventGroup $amazonFinancialEventGroupIds */
        $amazonFinancialEventGroupIds = AmazonFinancialEventGroup::query()
            ->where('integration_instance_id', $amazonIntegrationInstanceId)
            ->where('FinancialEventGroupId', $FinancialEventGroupId)
            ->first();

        return $amazonFinancialEventGroupIds;
    }

    public function getLastStartedDate(AmazonIntegrationInstance $amazonIntegrationInstance): ?string
    {
        return AmazonFinancialEventGroup::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->latest('FinancialEventGroupStart')
            ->pluck('FinancialEventGroupStart')
            ->first();
    }

    public function getClosedWithoutEvents(AmazonIntegrationInstance $amazonIntegrationInstance): EloquentCollection
    {
        return AmazonFinancialEventGroup::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('ProcessingStatus', AmazonFinancialEventGroupProcessingStatusEnum::Closed)
            ->whereDoesntHave('amazonFinancialEvents')
            ->orderBy('FinancialEventGroupStart', 'desc')
            ->get();
    }

    public function getFinancialEventGroupForSettlementReport(
        AmazonIntegrationInstance $amazonIntegrationInstance,
        array $settlementReportHeader
    ): ?AmazonFinancialEventGroup {
        return AmazonFinancialEventGroup::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('FinancialEventGroupStart', Carbon::parse($settlementReportHeader['settlement_start_date'])->toIso8601ZuluString())
            ->where('FinancialEventGroupEnd', Carbon::parse($settlementReportHeader['settlement_end_date'])->toIso8601ZuluString())
            ->where('OriginalTotalCurrencyAmount', floatval($settlementReportHeader['total_amount']))
            ->where('OriginalTotalCurrencyCode', $settlementReportHeader['currency'])
            ->first();
    }

    /**
     * Returns amazon financial event groups that don't yet have accounting transactions and satisfy the conditions
     * for the batchable sales order invoice to be created for it.
     */
    public function getFinancialEventGroupsNeedingAccountingTransactions(AmazonIntegrationInstance $amazonIntegrationInstance, array $amazonFinancialEventGroupIds): EloquentCollection
    {
        $status = AmazonFinancialEventGroupProcessingStatusEnum::Closed->value;

        $sqlCondition = <<<SQL
	`integration_instance_id` = {$amazonIntegrationInstance->id}
	AND `ProcessingStatus` = "$status"
-- 	Not linked to existing accounting transaction
	AND NOT EXISTS (
		SELECT
			*
		FROM
			`accounting_transactions` at
		WHERE
			afeg.`id` = at.`link_id`
			AND at.`link_type` = "Modules\\\Amazon\\\Entities\\\AmazonFinancialEventGroup"
	)
-- 	# Settlement data has all the required behavior
-- 	# Has sales orders, which have accounting transactions that are batchable
-- 	# Has nominal code mappings
	AND EXISTS (
		SELECT
			*
		FROM
			`amazon_report_settlement_data` arsd
			INNER JOIN amazon_orders ao ON ao.integration_instance_id = arsd.integration_instance_id AND ao.AmazonOrderId = arsd.order_id
			INNER JOIN sales_orders so ON so.sales_channel_order_id = ao.id
				AND so.sales_channel_order_type = "Modules\\\Amazon\\\Entities\\\AmazonOrder"
			INNER JOIN accounting_transactions at ON at.link_id = so.id
				AND at.link_type = "App\\\Models\\\SalesOrder"
				AND at.is_batchable = 1
		WHERE
			afeg.`id` = arsd. `amazon_financial_event_group_id`
			AND EXISTS (
				SELECT
					*
				FROM
					`amazon_report_settlement_type_mappings` arstm
					INNER JOIN nominal_codes nc ON nc.id = arstm.nominal_code_id
				WHERE
					arsd. `integration_instance_id` = arstm. `integration_instance_id`
					AND arsd. `transaction_type` = arstm. `transaction_type`
					AND arsd. `amount_type` = arstm. `amount_type`
					AND arsd. `amount_description` = arstm. `amount_description`)
	) 
-- 	# Settlement data has amazon orders created for any order related line (order_id != "")
	AND NOT EXISTS (
		SELECT
			*
		FROM
			`amazon_report_settlement_data` arsd 
			LEFT JOIN amazon_orders ao ON ao.AmazonOrderId = arsd.order_id
		WHERE
			afeg.`id` = arsd.`amazon_financial_event_group_id`
			AND arsd.`order_id` != ""
			AND ao.id IS NULL
	)
SQL;

        $query = AmazonFinancialEventGroup::query()
            ->from('amazon_financial_event_groups as afeg')
            ->select('afeg.*')
            ->whereRaw($sqlCondition);

        if (! empty($amazonFinancialEventGroupIds)) {
            $query->whereIn('id', $amazonFinancialEventGroupIds);
        }

        return $query->get();
    }

    private function baseAccountingStatusUpdateQuery(AmazonIntegrationInstance $amazonIntegrationInstance, array $amazonFinancialEventGroupIds): Builder
    {
        $query = AmazonFinancialEventGroup::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereDoesntHave('accountingTransaction')
            ->where('ProcessingStatus', AmazonFinancialEventGroupProcessingStatusEnum::Closed);

        if (! empty($amazonFinancialEventGroupIds)) {
            $query->whereIn('id', $amazonFinancialEventGroupIds);
        }

        return $query;
    }

    public function updateAccountingStatusForGroupsMissingSettlementData(AmazonIntegrationInstance $amazonIntegrationInstance, array $amazonFinancialEventGroupIds): void
    {
        $this->baseAccountingStatusUpdateQuery($amazonIntegrationInstance, $amazonFinancialEventGroupIds)
            ->whereDoesntHave('amazonReportSettlementData')
            ->update(['accounting_status' => AmazonFinancialEventGroupAccountingStatusEnum::STATUS_MISSING_SETTLEMENT_DATA]);
    }

    public function updateAccountingStatusForGroupsMissingSettlementDataMapping(AmazonIntegrationInstance $amazonIntegrationInstance, array $amazonFinancialEventGroupIds): void
    {
        $this->baseAccountingStatusUpdateQuery($amazonIntegrationInstance, $amazonFinancialEventGroupIds)
            ->whereHas('amazonReportSettlementData', function ($query) {
                $query->whereDoesntHave('amazonReportSettlementTypeMapping', function ($query) {
                    $query->whereHas('nominalCode');
                });
            })
            ->update(['accounting_status' => AmazonFinancialEventGroupAccountingStatusEnum::STATUS_MISSING_MAPPINGS]);
    }

    public function updateAccountingStatusForGroupsMissingSettlementDataAmazonOrder(AmazonIntegrationInstance $amazonIntegrationInstance, array $amazonFinancialEventGroupIds): void
    {
        $this->baseAccountingStatusUpdateQuery($amazonIntegrationInstance, $amazonFinancialEventGroupIds)
            ->where('ProcessingStatus', AmazonFinancialEventGroupProcessingStatusEnum::Closed)
            ->whereHas('amazonReportSettlementData', function ($query) {
                $query->where('order_id', '!=', '');
                $query->whereDoesntHave('amazonOrder');
            })
            ->update(['accounting_status' => AmazonFinancialEventGroupAccountingStatusEnum::STATUS_MISSING_AMAZON_ORDER]);
    }

    public function updateAccountingStatusForGroupsMissingSettlementDataSalesOrder(AmazonIntegrationInstance $amazonIntegrationInstance, array $amazonFinancialEventGroupIds): void
    {
        $this->baseAccountingStatusUpdateQuery($amazonIntegrationInstance, $amazonFinancialEventGroupIds)
            ->whereHas('amazonReportSettlementData', function ($query) {
                $query->where('order_id', '!=', '');
                $query->whereHas('amazonOrder', function ($query) {
                    $query->whereDoesntHave('salesOrder');
                });
            })
            ->update(['accounting_status' => AmazonFinancialEventGroupAccountingStatusEnum::STATUS_MISSING_SALES_ORDER]);
    }

    public function updateAccountingStatusForGroupsMissingSettlementDataSalesOrderInvoice(AmazonIntegrationInstance $amazonIntegrationInstance, array $amazonFinancialEventGroupIds): void
    {
        $this->baseAccountingStatusUpdateQuery($amazonIntegrationInstance, $amazonFinancialEventGroupIds)
            ->whereHas('amazonReportSettlementData', function ($query) {
                $query->where('order_id', '!=', '');
                $query->whereHas('amazonOrder', function ($query) {
                    $query->whereHas('salesOrder', function ($query) {
                        $query->whereDoesntHave('accountingTransaction');
                    });
                });
            })
            ->update(['accounting_status' => AmazonFinancialEventGroupAccountingStatusEnum::STATUS_MISSING_SALES_ORDER_INVOICE]);
    }

    public function completeAccountingTransactionLink(AmazonFinancialEventGroup $amazonFinancialEventGroup, AccountingTransaction $accountingTransaction): void
    {
        $amazonFinancialEventGroup->amazonReportSettlementData()->each(function (AmazonReportSettlementData $settlementData) use ($accountingTransaction) {
            /// Update the batchable sales order invoices to link to the batch sales order invoice (using parent_id) without triggering timestamp update
            if ($salesOrderInvoice = $settlementData->amazonOrder?->salesOrder?->accountingTransaction) {
                $salesOrderInvoice->update(['parent_id' => $accountingTransaction->id], ['touch' => false]);
            }
        });

        $amazonFinancialEventGroup->update(['accounting_status' => AmazonFinancialEventGroupAccountingStatusEnum::STATUS_DONE]);
    }

    public function seedTypeMappings(AmazonIntegrationInstance $amazonIntegrationInstance): void
    {
        $new_mappings = AmazonReportSettlementData::query()->leftJoin('amazon_report_settlement_type_mappings', function ($join) {
            $join->on('amazon_report_settlement_data.integration_instance_id', '=', 'amazon_report_settlement_type_mappings.integration_instance_id')
                ->on('amazon_report_settlement_data.transaction_type', '=', 'amazon_report_settlement_type_mappings.transaction_type')
                ->on('amazon_report_settlement_data.amount_type', '=', 'amazon_report_settlement_type_mappings.amount_type')
                ->on('amazon_report_settlement_data.amount_description', '=', 'amazon_report_settlement_type_mappings.amount_description');
        })
            ->whereNull('amazon_report_settlement_type_mappings.id')
            ->where('amazon_report_settlement_data.integration_instance_id', $amazonIntegrationInstance->id)
            ->get();

        foreach ($new_mappings as $mapping) {
            $new_mapping = new AmazonReportSettlementTypeMapping();
            $new_mapping->integration_instance_id = $mapping->integration_instance_id;
            $new_mapping->transaction_type = $mapping->transaction_type;
            $new_mapping->amount_type = $mapping->amount_type;
            $new_mapping->amount_description = $mapping->amount_description;
            $new_mapping->save();
        }
    }
}
