<?php

namespace Modules\Amazon\Repositories;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Modules\Amazon\Data\AmazonCreateFeedData;
use Modules\Amazon\Data\AmazonFeedData;
use Modules\Amazon\Entities\AmazonFeedRequest;
use Modules\Amazon\Entities\AmazonFeedSubmission;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Enums\Entities\AmazonFeedProcessingStatusEnum;
use Modules\Amazon\Enums\Entities\AmazonFeedTypeEnum;

class AmazonFeedRepository
{
    public function requestExists(int $integrationInstanceId, AmazonFeedTypeEnum $feed_type): bool
    {
        return (bool) AmazonFeedRequest::query()
            ->where('integration_instance_id', $integrationInstanceId)
            ->where('feedType', $feed_type())->count();
    }

    public function saveRequest(
        int $integration_instance_id,
        AmazonFeedTypeEnum $feed_type,
        ?array $options = null,
        ?array $metadata = null
    ): Model|AmazonFeedRequest {
        //Added specifically for POST_ORDER_ACKNOWLEDGEMENT_DATA
        if (! is_null($metadata)) {
            return AmazonFeedRequest::create([
                'integration_instance_id' => $integration_instance_id,
                'feedType' => $feed_type,
                'feedOptions' => $options,
                'metadata' => $metadata,
            ]);
        }

        return AmazonFeedRequest::query()->updateOrCreate([
            'integration_instance_id' => $integration_instance_id,
            'feedType' => $feed_type,
        ], [
            'feedOptions' => $options,
            'metadata' => $metadata,
        ]);
    }

    /**
     * Get Amazon requests feeds.
     */
    public function getFeedRequests(AmazonIntegrationInstance $amazonIntegrationInstance): EloquentCollection
    {
        return AmazonFeedRequest::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->get();
    }

    public function getFeedRequestFromId(int $id): AmazonFeedRequest|Model
    {
        return AmazonFeedRequest::query()->findOrFail($id);
    }

    public function create(AmazonFeedRequest $amazonFeedRequest, AmazonFeedData|AmazonCreateFeedData $feed): AmazonFeedSubmission|Model
    {
        $feedSubmission = new AmazonFeedSubmission();

        $feedSubmission->fill(array_merge([
            'integration_instance_id' => $amazonFeedRequest->integration_instance_id,
        ], $feed->toArray()));

        $feedSubmission->feedType = $amazonFeedRequest->feedType;

        if ($amazonFeedRequest->feedType === AmazonFeedTypeEnum::ORDER_FULFILLMENT) {
            $feedSubmission->metadata = $amazonFeedRequest->metadata;
        }

        $feedSubmission->save();

        return $feedSubmission;
    }

    public function getFromFeedId(string $feedId): ?AmazonFeedSubmission
    {
        return AmazonFeedSubmission::where('feedId', $feedId)->first();
    }

    public function getPendingFeedSubmissions(AmazonIntegrationInstance $amazonIntegrationInstance): EloquentCollection
    {
        return AmazonFeedSubmission::where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where(function (Builder $query) {
                $query->whereNull('processingStatus')
                    ->orWhereIn('processingStatus', AmazonFeedProcessingStatusEnum::RETRYABLE_STATUSES);
            })
            ->get();
    }

    public function getPendingFeedSubmission(AmazonIntegrationInstance $amazonIntegrationInstance, AmazonFeedTypeEnum $amazonFeedType): ?AmazonFeedRequest
    {
        return AmazonFeedRequest::where([
            'feedType' => $amazonFeedType(),
            'integration_instance_id' => $amazonIntegrationInstance->id,
        ])->first();
    }

    public function appendAmazonOrderToAmazonFulfillmentFeed(AmazonFeedRequest $amazonFeedRequest, Collection $salesOrderCollection): AmazonFeedRequest
    {
        //Make metadata
        $salesOrderCollection->each(function ($dto) use ($amazonFeedRequest) {
            $metadata = collect($amazonFeedRequest->metadata);
            $existingAmazonOrder = $metadata->where('AmazonOrderID', $dto->AmazonOrderID);

            if ($existingAmazonOrder->count() > 0) {
                $_existingAmazonOrder = $existingAmazonOrder->first();
                $existingAmazonOrderItems = collect($_existingAmazonOrder['Items']);

                collect($dto->Items)
                    ->each(function ($dtoItem) use (&$existingAmazonOrderItems) {
                        if (! $existingAmazonOrderItems->where('checksum', $dtoItem->checksum)->first()) {
                            $existingAmazonOrderItems->push($dtoItem->toArray());
                        }
                    });
                $_existingAmazonOrder['Items'] = $existingAmazonOrderItems->toArray();

                $metadata[$existingAmazonOrder->keys()->first()] = $_existingAmazonOrder;

                $amazonFeedRequest->metadata = $metadata->toArray();
            } else {
                $amazonFeedRequest->metadata = collect($amazonFeedRequest->metadata)->push($dto->toArray());
            }
            $amazonFeedRequest->update();
        });

        return $amazonFeedRequest;
    }
}
