<?php

namespace Modules\Amazon\Repositories;

use App\Abstractions\Integrations\AbstractIntegrationRepository;
use Illuminate\Support\Collection;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonNewFbaInboundShipment;
use Modules\Amazon\Enums\Entities\FbaInboundShipmentStatusEnum;
use Modules\Amazon\Entities\AmazonFbaInboundShipFromMapping;

class AmazonFbaInboundShipmentRepository extends AbstractIntegrationRepository
{
    public function getModelClassName(): string
    {
        return AmazonFbaInboundShipment::class;
    }

    public function getActiveItemlessShipmentIds(int $limit = 30): Collection
    {
        return AmazonFbaInboundShipment::query()
            ->whereIn('ShipmentStatus', FbaInboundShipmentStatusEnum::STATUSES_ACTIVE)
            ->whereDoesntHave('amazonFbaInboundShipmentItems')
            ->limit($limit)
            ->orderBy('created_at')
            ->pluck('ShipmentId');
    }

    public function getLastUpdateDate(): ?string
    {
        return AmazonFbaInboundShipment::query()->latest('updated_at')->pluck('updated_at')->first();
    }

    public function getFromShipmentId(AmazonIntegrationInstance $amazonIntegrationInstance, string $ShipmentId): ?AmazonFbaInboundShipment
    {
        /** @var AmazonFbaInboundShipment $amazonFbaInboundShipment */
        $amazonFbaInboundShipment = AmazonFbaInboundShipment::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->where('ShipmentId', $ShipmentId)
            ->first();

        return $amazonFbaInboundShipment;
    }

    public function getUnlinkedShipments(AmazonIntegrationInstance $amazonIntegrationInstance): Collection
    {
        return AmazonFbaInboundShipment::query()
            ->select(['id', 'ShipmentId'])
            ->selectRaw('CONCAT(ShipmentId, " (", ShipFromName, ") - ", ShipmentName) as description')
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereHas('amazonFbaInboundShipmentItems')
            ->whereNull('sku_link_id')
            ->whereIn('ShipmentStatus', FbaInboundShipmentStatusEnum::STATUSES_ACTIVE)
            ->get();
    }

    public function getUnprocessedWithMappingOfType(AmazonIntegrationInstance $amazonIntegrationInstance, string $type): Collection
    {
        $mappings = AmazonFbaInboundShipFromMapping::where('link_type', $type)
            ->get();

        return AmazonFbaInboundShipment::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereHas('amazonFbaInboundShipmentItems')
            ->whereNull('sku_link_id')
            ->where('is_before_initial_count', false)
            ->whereIn('ShipFromName', $mappings->pluck('name'))
            ->get();
    }

    public function getUnprocessedCancellations(AmazonIntegrationInstance $amazonIntegrationInstance): Collection
    {
        return AmazonFbaInboundShipment::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereNotNull('sku_link_id')
            ->whereIn('ShipmentStatus', [
                FbaInboundShipmentStatusEnum::CANCELLED,
                FbaInboundShipmentStatusEnum::DELETED,
            ])
            ->get();
    }

    public function duplicatedByNewInbounds(AmazonIntegrationInstance $amazonIntegrationInstance, Collection $shipmentIds): Collection
    {
        return AmazonNewFbaInboundShipment::query()
            ->where('integration_instance_id', $amazonIntegrationInstance->id)
            ->whereIn('shipmentConfirmationId', $shipmentIds)
            ->pluck('shipmentConfirmationId');
    }
}
