<?php

namespace Modules\Amazon\Managers;

use App\Exceptions\StoredFileDoesNotExistException;
use App\Helpers;
use App\Models\NominalCode;
use Illuminate\Support\Facades\Storage;
use Modules\Amazon\Data\AmazonReportSettlementTypeMappingsData;
use Modules\Amazon\Entities\AmazonReportSettlementTypeMapping;
use Modules\Amazon\Exceptions\AmazonAccountingMappingFileEmptyException;

class AmazonReportSettlementDataManager
{
    public function importSettlementTypes(int $integrationInstanceId, string $storedFilename): void
    {
        if (!Storage::disk('model-imports')->exists($storedFilename)) {
            throw new StoredFileDoesNotExistException();
        }

        $file = Storage::disk('model-imports')->path($storedFilename);

        $data = Helpers::csvFileToCollection($file);

        Storage::disk('model-imports')->delete($storedFilename);

        if ($data->count() === 0) {
            throw new AmazonAccountingMappingFileEmptyException();
        }

        $settlementTypesLineData = $data->map(fn(
            $item,
        ) => $this->mapTypeMappingLine($integrationInstanceId, $item));

        /** @var AmazonReportSettlementTypeMappingsData $row */
        foreach ($settlementTypesLineData as $row) {
            AmazonReportSettlementTypeMapping::where('integration_instance_id', $row->integration_instance_id)
                ->where('financial_code', $row->financial_code)
                ->update(['nominal_code_id' => $row->nominal_code_id]);
        }
    }

    private function mapTypeMappingLine(int $integrationInstanceId, array $item): AmazonReportSettlementTypeMappingsData
    {
        return AmazonReportSettlementTypeMappingsData::from([
            'integration_instance_id' => $integrationInstanceId,
            'financial_code' => $item['Financial Code'],
            'nominal_code_id' => strlen($item['Nominal Code']) ? $this->extractNominalCode($item['Nominal Code']) : null,
        ]);
    }

    private function extractNominalCode(string $code): ?int
    {
        $code = explode(' - ', $code);
        if (intval($code[0])) {
            return NominalCode::where('code', $code[0])->value('id');
        }

        $nominalCode = NominalCode::where('name', $code[0])->get();

        if (count($nominalCode) == 1) {
            return $nominalCode->first()->id;
        }
        return null;
    }
}