<?php

namespace Modules\Amazon\Managers;

use App;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelManager;
use App\Managers\WarehouseTransferManager;
use App\Repositories\InventoryAdjustmentRepository;
use App\Repositories\SalesCreditRepository;
use App\Repositories\SalesOrderFulfillmentRepository;
use App\Repositories\WarehouseTransferRepository;
use Carbon\Carbon;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Facades\Bus;
use Illuminate\Support\Facades\Storage;
use Modules\Amazon\Actions\RequestAmazonReport;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetReportsAdt;
use Modules\Amazon\Data\AmazonReportData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Data\RequestAmazonReportData;
use Modules\Amazon\Entities\AmazonFbaReportCustomerReturn;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Entities\AmazonReportSettlementData;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Jobs\AmazonInitializeSkuProductsJob;
use Modules\Amazon\Jobs\CreateAmazonFnSkusToSkuProductsJob;
use Modules\Amazon\Jobs\CreateAmazonSearchCatalogItemsJobsJob;
use Modules\Amazon\Jobs\CreateAuditTrailFromAmazonLedgerJob;
use Modules\Amazon\Jobs\LinkLedgersToDetailReportsJob;
use Modules\Amazon\Jobs\SeedAmazonReportSettlementDataTypeMappingsJob;
use Modules\Amazon\Jobs\SyncAmazonReportsJob;
use Modules\Amazon\Repositories\AmazonFbaInboundShipmentRepository;
use Modules\Amazon\Repositories\AmazonFnskuRepository;
use Modules\Amazon\Repositories\AmazonLedgerRepository;
use Modules\Amazon\Repositories\AmazonOrderItemRepository;
use Modules\Amazon\Repositories\AmazonOrderRepository;
use Modules\Amazon\Repositories\AmazonProductRepository;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Modules\Amazon\Services\AmazonClient;
use SellingPartnerApi\ApiException;
use Throwable;

class AmazonReportManager extends AbstractSalesChannelManager
{
    private AmazonReportRepository $reportRepository;

    /**
     * @throws Exception
     */
    public function __construct(private readonly AmazonIntegrationInstance $amazonIntegrationInstance)
    {
        parent::__construct($this->amazonIntegrationInstance, new AmazonClient($this->amazonIntegrationInstance));
        $this->reportRepository = app(AmazonReportRepository::class);
        $this->amazonOrderRepository = app(AmazonOrderRepository::class);
        $this->amazonOrderItemRepository = app(AmazonOrderItemRepository::class);
        $this->amazonProductRepository = app(AmazonProductRepository::class);
        $this->ledgers = app(AmazonLedgerRepository::class);
        $this->fnskus = app(AmazonFnskuRepository::class);
        $this->inventoryAdjustmentRepository = app(InventoryAdjustmentRepository::class);
        $this->salesCreditRepository = app(SalesCreditRepository::class);
        $this->salesOrderFulfillmentRepository = app(SalesOrderFulfillmentRepository::class);
        $this->amazonFbaInboundShipmentRepository = app(AmazonFbaInboundShipmentRepository::class);
        $this->warehouseTransferRepository = app(WarehouseTransferRepository::class);
        $this->warehouseTransferProcessor = app(WarehouseTransferManager::class);
    }

    /**
     * Get report_document_id and download report file.
     *
     * @throws ApiException
     * @throws GuzzleException
     * @throws Throwable
     */
    public function sync(AmazonReport $amazonReport): void
    {
        if (is_null($amazonReport->reportDocumentId)) {
            $this->get($amazonReport);
            $amazonReport = $amazonReport->fresh();
        }

        if (! is_null($amazonReport->reportDocumentId) && is_null($amazonReport->filename)) {
            $this->download($amazonReport);
        }
    }

    /**
     * @throws ApiException
     * @throws Throwable
     */
    private function get(AmazonReport $amazonReport): void
    {
        $amazonReportDto = $this->client->getReportById($amazonReport->reportId);
        $amazonReportDto->integration_instance_id = $amazonReport->integration_instance_id;
        $this->reportRepository->saveOne($amazonReport, $amazonReportDto);
    }

    /**
     * @throws GuzzleException
     * @throws Exception
     * @throws Throwable
     */
    private function download(AmazonReport $amazonReport): void
    {
        customlog('amazon', 'Downloading Amazon Report: '.$amazonReport->reportType->value.'('.$amazonReport->reportId.')');
        $documentResult = $this->client->getReportDocument($amazonReport->reportDocumentId);
        $filename = $amazonReport->reportId.'.tsv';

        Storage::disk('amazon_reports')->put($filename, $documentResult);

        $amazonReport->updateFilename($filename);

        $this->process($amazonReport);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function process(AmazonReport $amazonReport): void
    {
        customlog('amazon', 'Processing Amazon Report: '.$amazonReport->reportType->value.'('.$amazonReport->reportId.')');
        switch (($amazonReport->reportType)()) {
            // FBA Detail Reports
            case AmazonReportTypeEnum::FBA_REPORT_CUSTOMER_RETURNS():
            case AmazonReportTypeEnum::FBA_REPORT_REMOVAL_SHIPMENTS():
            case AmazonReportTypeEnum::FBA_REPORT_SHIPMENTS():
                $this->reportRepository->saveReport($amazonReport);
                Bus::dispatchChain(
                    new LinkLedgersToDetailReportsJob($this->amazonIntegrationInstance),
                    new CreateAuditTrailFromAmazonLedgerJob($this->amazonIntegrationInstance, $amazonReport)
                );
                break;
                // FBA Ledger Report
            case AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER():
                $this->reportRepository->saveReport($amazonReport);
                if (App::environment() != 'testing') {
                    Bus::dispatchChain(
                        new CreateAmazonFnSkusToSkuProductsJob($this->amazonIntegrationInstance),
                        new LinkLedgersToDetailReportsJob($this->amazonIntegrationInstance),
                        new CreateAuditTrailFromAmazonLedgerJob($this->amazonIntegrationInstance, $amazonReport)
                    );
                }
                break;
                // Settlement Report
            case AmazonReportTypeEnum::SETTLEMENT_REPORT():
                $this->reportRepository->saveReport($amazonReport);
                dispatch(new SeedAmazonReportSettlementDataTypeMappingsJob($this->amazonIntegrationInstance));
                break;
                // Other FBA Reports
            case AmazonReportTypeEnum::FBA_INVENTORY():
            case AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS():
            case AmazonReportTypeEnum::FBA_REPORT_RESTOCK():
                $this->reportRepository->saveReport($amazonReport);
                break;
            case AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY():
                $this->reportRepository->saveReport($amazonReport);
                Bus::dispatchChain(
                    new CreateAmazonFnSkusToSkuProductsJob($this->amazonIntegrationInstance, $amazonReport),
                    new AmazonInitializeSkuProductsJob($this->amazonIntegrationInstance)
                );
                break;
                // Products
            case AmazonReportTypeEnum::PRODUCTS():
                (new AmazonProductManager($amazonReport->integrationInstance))->import($amazonReport);
                dispatch(new CreateAmazonSearchCatalogItemsJobsJob($this->amazonIntegrationInstance))->onQueue('sales-channels');
                break;
            default:
                break;
        }
        $amazonReport->processed_at = Carbon::now();
        $amazonReport->save();
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function requestCreateSync(
        AmazonReportTypeEnum $reportType,
        ?Carbon $data_start_date = null,
        ?Carbon $data_end_date = null,
        ?array $options = []
    ): void {
        $reportRequest = (new RequestAmazonReport(RequestAmazonReportData::from([
                'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
                'report_type' => $reportType,
                'data_start_date' => $data_start_date,
                'data_end_date' => $data_end_date,
                'options' => $options
            ])))->handle();

        // Could be multiple reports if it is splittable
        $reports = $reportRequest->createdReports;

        foreach ($reports as $report) {
            $this->sync($report);
        }
    }

    /**
     * @throws Exception
     */
    public function refreshSettlementReports(AmazonGetReportsAdt $parameters): AmazonResponseData
    {
        customlog('amazon', 'Refreshing Settlement Reports');
        $parameters->start = $parameters->start ?? $this->reportRepository->getStartDateForNew(AmazonReportSettlementData::class);
        $parameters->reportType = AmazonReportTypeEnum::SETTLEMENT_REPORT;

        $amazonResponseDto = $this->client->getReports($parameters);

        $amazonReportCollection = $amazonResponseDto->collection;

        if (! $amazonReportCollection->count()) {
            return $amazonResponseDto;
        }

        $amazonReportCollection = $amazonReportCollection->map(function (AmazonReportData $amazonReportData) {
            $amazonReportData->integration_instance_id = $this->amazonIntegrationInstance->id;
            $amazonReportData->marketplaceIds = json_encode($amazonReportData->marketplaceIds);
            return $amazonReportData;
        });

        $this->reportRepository->save($amazonReportCollection->toCollection(), AmazonReport::class);

        SyncAmazonReportsJob::dispatch($this->amazonIntegrationInstance);

        return $amazonResponseDto;
    }

    public function getCustomerReturnFromLpn(string $lpn): ?AmazonFbaReportCustomerReturn
    {
        return $this->reportRepository->getCustomerReturnFromLpn($this->amazonIntegrationInstance, $lpn);
    }
}
