<?php

namespace Modules\Amazon\Managers;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelOrderManager;
use App\Models\SalesOrderFulfillment;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Modules\Amazon\ApiDataTransferObjects\AmazonFulfillOrdersAdt;
use Modules\Amazon\Data\AmazonCreateOrderFulfillmentFeedData;
use Modules\Amazon\Data\AmazonCreateOrderFulfillmentItemData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Entities\AmazonFeedRequest;
use Modules\Amazon\Entities\AmazonFeedSubmission;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonOrder;
use Modules\Amazon\Enums\Entities\AmazonFeedTypeEnum;
use Modules\Amazon\Exceptions\AmazonFeedException;
use Modules\Amazon\Jobs\CreateAmazonRefreshOrderItemsJobs;
use Modules\Amazon\Jobs\GenerateCreateSkuOrdersJobs;
use Modules\Amazon\Repositories\AmazonFeedRepository;
use Modules\Amazon\Repositories\AmazonOrderItemRepository;
use Modules\Amazon\Repositories\AmazonOrderRepository;
use Modules\Amazon\Services\AmazonClient;
use Spatie\LaravelData\DataCollection;

class AmazonOrderManager extends AbstractSalesChannelOrderManager
{
    private AmazonOrderItemRepository $amazonOrderItemRepository;

    private AmazonFeedRepository $amazonFeedRepository;

    /**
     * @throws Exception
     */
    public function __construct(protected AmazonIntegrationInstance $amazonIntegrationInstance)
    {
        $this->amazonOrderItemRepository = app(AmazonOrderItemRepository::class);
        $this->amazonFeedRepository = app(AmazonFeedRepository::class);

        $this->setModel(new AmazonOrder());

        parent::__construct(
            $amazonIntegrationInstance,
            new AmazonClient($amazonIntegrationInstance),
            app(AmazonOrderRepository::class)
        );
    }

    protected function getOrderRepository()
    {
        return $this->orderRepository;
    }

    protected function getOrderItemRepository()
    {
        return $this->amazonOrderItemRepository;
    }

    protected function postProcess(?string $orderId = null): void
    {
        //customlog('amazon', 'Post Processing for Amazon Order: '.($orderId ?? 'orderId not specified'));
        if (is_null($orderId)) {
            // Create jobs to refresh order items.
            CreateAmazonRefreshOrderItemsJobs::dispatch($this->amazonIntegrationInstance);
        } else {
            customlog('amazon', 'Getting order items for Amazon Order: '.$orderId);
            $this->getOrderItems($orderId);
        }
    }

    public function postProcessItems(): void
    {
        dispatch(new GenerateCreateSkuOrdersJobs($this->amazonIntegrationInstance))->onQueue('salesOrderProcessing');
    }

    protected function fetchOrderItems(string $orderId): AmazonResponseData
    {
        return $this->client->getOrderItems($orderId);
    }

    public function getOrderItems(string $orderId): AmazonResponseData
    {
        $orderItems = $this->fetchOrderItems($orderId);

        $order = $this->orderRepository->getOrderFromId($this->integrationInstance, $orderId);

        $this->getOrderItemRepository()->save($order->id, $orderItems->collection);

        return $orderItems;
    }

    public function refreshOrderItems(array $orderIds): void
    {
        foreach ($orderIds as $orderId) {
            $this->getOrderItems($orderId);
        }
        $this->postProcessItems();
    }

    /**
     * @throws Exception
     */
    public function fulfillOrdersUsingAmazonFeed(AmazonFeedRequest $amazonFeedRequest)
    {
        $apo = new AmazonFulfillOrdersAdt(
            AmazonCreateOrderFulfillmentFeedData::collection($amazonFeedRequest->metadata),
        );

        try {
            $feedResponse = $this->client->fulfillOrders($apo, $amazonFeedRequest);
            $amazonFeedRequest->requestFeedDocumentXml = $feedResponse->requestFeedDocumentXml;
            $amazonFeedRequest->update();

        } catch (AmazonFeedException $feedException) {
            $amazonFeedRequest->salesOrderFulfillments->each(function ($salesOrderFulfillment) {
                $this->salesOrderFulfillmentRepository->markAsNotSubmittedToSalesChannel($salesOrderFulfillment);
            });
            throw $feedException;
        }

        // Create Amazon Feed
        $amazonFeedSubmission = $this->amazonFeedRepository->create(
            $amazonFeedRequest,
            $feedResponse
        );

        // Link amazon feed submission
        $amazonFeedRequest->salesOrderFulfillments->each(function (SalesOrderFulfillment $salesOrderFulfillment) use ($amazonFeedSubmission) {
            $this->salesOrderFulfillmentRepository->markAsSubmittedToSalesChannel($salesOrderFulfillment, $amazonFeedSubmission);
        });

        return $amazonFeedSubmission;
    }

    /**
     * Amazon fulfillment uses feeds, so fulfillOrders is overridden
     */
    public function fulfillOrders(ApiDataTransformerInterface|AmazonFulfillOrdersAdt $parameters): AmazonFeedSubmission|AmazonFeedRequest // TODO: @Jatin: This method is returning an inconsistent type
    {
        if ($parameters->amazonCreateOrderFulfillmentFeedDtoCollection->isEmpty()) {
            throw new Exception('No orders are there to fulfill.', 1);
        }

        //Create or update feed.
        $existingFeed = $this->amazonFeedRepository
            ->getPendingFeedSubmission(
                $this->amazonIntegrationInstance,
                AmazonFeedTypeEnum::ORDER_FULFILLMENT
            );

        if ($existingFeed) {
            $this->amazonFeedRepository->appendAmazonOrderToAmazonFulfillmentFeed($existingFeed, $parameters->amazonCreateOrderFulfillmentFeedDtoCollection);
        } else {
            $existingFeed = $this->amazonFeedRepository->saveRequest(
                $this->amazonIntegrationInstance->id,
                AmazonFeedTypeEnum::ORDER_FULFILLMENT,
                [],
                $parameters->amazonCreateOrderFulfillmentFeedDtoCollection->toArray()
            );
        }

        return $existingFeed;
    }

    /**
     * @throws Exception
     */
    public function fulfillPendingAmazonMfnOrders(): void
    {
        $collection = collect();

        $salesOrderFulfillments = $this->salesOrderFulfillmentRepository
            ->getPendingSalesOrderFulfillments($this->amazonIntegrationInstance)
            ->each(function ($salesOrderFulfillment) use (&$collection) {
                $amazonOrder = $salesOrderFulfillment->salesOrder->sales_channel_order;

                $carrierName = $salesOrderFulfillment->fulfilledShippingMethod ? $salesOrderFulfillment->fulfilledShippingMethod->shippingCarrier->name : $salesOrderFulfillment->fulfilled_shipping_method;

                $lines = [];

                $salesOrderFulfillment->salesOrderFulfillmentLines->each(function ($fulfillmentLine) use (&$lines, $salesOrderFulfillment) {
                    $salesOrderLine = $fulfillmentLine->salesOrderLine;

                    if ($bundle = $salesOrderLine->bundle) {
                        $quantity = $fulfillmentLine->quantity;

                        // Refer to ShopifySubmitTrackingInfo.php "//Bundle handling"

                        // get all fulfillment lines matching the bundle id
                        $fulfillmentLinesFromBundle = $salesOrderFulfillment->salesOrderFulfillmentLines->filter(function ($fulfillmentLine) use ($bundle) {
                            return $fulfillmentLine->salesOrderLine->bundle_id == $bundle->id;
                        });

                        /*
                     * Each of these fulfillment lines for this particular fulfillment equate to a single shopify fulfillment line, so we need to figure
                     * out the quantity of the bundle that was fulfilled
                     */
                        $bottleneckQuantity = $quantity;
                        $fulfillmentLinesFromBundle->each(function ($fulfillmentLine) use ($bundle, &$bottleneckQuantity, &$fulfillmentLinesIdsAlreadyProcessed) {
                            // get the component quantity
                            $component = $bundle->components->where('id', $fulfillmentLine->salesOrderLine->product->id)->first();

                            $componentQuantity = max(1, $component->pivot->quantity);

                            // get the quantity of the fulfillment line
                            $fulfilledQuantity = $fulfillmentLine->quantity;

                            // get the quantity of the fulfillment line divided by the component quantity
                            $quantityOfBundleShippedFromComponent = max(1, floor($fulfilledQuantity / $componentQuantity));

                            // set the quantity of the fulfillment line to the quantity divided by the component quantity
                            $bottleneckQuantity = min($quantityOfBundleShippedFromComponent, $bottleneckQuantity);

                            // We need to make sure no other fulfillment lines get processed that were in fulfillment lines from bundle.
                            $fulfillmentLinesIdsAlreadyProcessed[] = $fulfillmentLine->id;
                        });

                        $quantity = $bottleneckQuantity;

                        $lines[] = AmazonCreateOrderFulfillmentItemData::from([
                            'AmazonOrderItemCode' => $fulfillmentLine->salesOrderLine->sales_channel_line_id,
                            'MerchantFulfillmentItemID' => $fulfillmentLine->id,
                            'Quantity' => $quantity,
                        ]);
                    } else {
                        $lines[] = AmazonCreateOrderFulfillmentItemData::from([
                            'AmazonOrderItemCode' => $fulfillmentLine->salesOrderLine->sales_channel_line_id,
                            'MerchantFulfillmentItemID' => $fulfillmentLine->id,
                            'Quantity' => $fulfillmentLine->quantity,
                        ]);
                    }
                });
                $collection->push(
                    AmazonCreateOrderFulfillmentFeedData::from([
                        'AmazonOrderID' => $amazonOrder->AmazonOrderId,
                        'MerchantFulfillmentID' => $salesOrderFulfillment->id,
                        'FulfillmentDate' => $salesOrderFulfillment->fulfilled_at->format('Y-m-d\TH:i:s\Z'),
                        'FulfillmentData_CarrierName' => $carrierName ?? 'Other',
                        'FulfillmentData_ShipperTrackingNumber' => $salesOrderFulfillment->tracking_number,
                        'Items' => $lines,
                    ])
                );
            });

        // This saves the feed request but does not submit it to Amazon
        if (! $collection->isEmpty()) {
            $amazonFeed = $this->fulfillOrders(new AmazonFulfillOrdersAdt(
                $collection,
            ));

            //We have used to set submitted_to_sales_channel_at = null if feed fails
            $salesOrderFulfillments->each(function (SalesOrderFulfillment $salesOrderFulfillment) use ($amazonFeed) {
                $this->salesOrderFulfillmentRepository->markAsSubmittedToSalesChannel($salesOrderFulfillment, $amazonFeed);
            });
        }

    }

    // This doesn't seem to be used and doesn't belong in the amazon order manager I believe... I've asked Jatin what it is for.
    //    public function mapShipping(string $shipServiceLevel)
    //    {
    //        $amazonOrders = $this->orderRepository->getSalesOrdersForShipping($this->amazonIntegrationInstance, $shipServiceLevel);
    //
    //        $salesOrders = SalesOrder::with([])
    //                                     ->whereIn('id', $amazonOrders->pluck('sku_sales_order_id'))
    //                                     ->get();
    //
    //        foreach ($salesOrders as $salesOrder) {
    //            /** @var AmazonOrder $order */
    //            $order = $amazonOrders->firstWhere('sku_sales_order_id', $salesOrder->id);
    //
    //            $skuOrderLines = collect($order->getSkuOrderLines());
    //
    //            $shippingMethodId = $order->getSkuShippingMethod();
    //
    //            $productLinesMapped = $skuOrderLines->where('is_product', true)
    //                                          ->where('warehouse_id', null)->isEmpty();
    //
    //            // map sales order shipping method
    //            $salesOrder->shipping_method_id = $shippingMethodId;
    //            $salesOrder->requested_shipping_method = $order->ShipServiceLevel;
    //            $salesOrder->save();
    //
    //            if ($productLinesMapped) {
    //                if (! $order->is_draft) {
    //                    $salesOrder->approve(ApproveSalesOrderService::APPROVE_DROPSHIP_CREATE_OR_UPDATE);
    //                }
    //
    //                if ($order->OrderStatus == OrderStatusEnum::STATUS_SHIPPED) {
    //                    $salesOrder->fullyFulfill();
    //                }
    //
    //                if ($order->OrderStatus == OrderStatusEnum::STATUS_PARTIALLY_SHIPPED) {
    //                    $order->partiallyFulfill($salesOrder);
    //                }
    //            }
    //        }
    //    }

    public function preOrdersProcess(DataCollection $orders): DataCollection
    {
        if ($integrationStartDate = Carbon::parse($this->integrationInstance->integration_settings['start_date'])) {
            $orders = $orders->filter(function ($order) use ($integrationStartDate) {
                if (is_null($order->json_object['PurchaseDate'])) {
                    return true;
                } elseif (Carbon::parse($order->json_object['PurchaseDate'])->lessThan($integrationStartDate)) {
                    return false;
                }

                return true;
            })
                ->values();
        }

        return $orders;
    }

    public function getSalesChannelOrdersQuery(
        array $orderIds = [],
        bool $createForAll = false,
        $enforceDateRestriction = true
    ): Builder {
        $builder = parent::getSalesChannelOrdersQuery($orderIds, $createForAll, $enforceDateRestriction);
        // Condition where SalesChannel doesn't start with Non-Amazon
        $builder->where('SalesChannel', 'not like', 'Non-Amazon%');
        return $builder;
    }
}
