<?php

namespace Modules\Amazon\Managers;

use App\Models\InventoryMovement;
use App\Services\InventoryManagement\InventoryEvent;
use Carbon\Carbon;
use Exception;
use Modules\Amazon\Data\AmazonLedgerForValuationData;
use Modules\Amazon\Data\FbaInventoryValuationData;
use Modules\Amazon\Data\FbaInventoryValuationFnskuProductData;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedgerSummary;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Repositories\AmazonLedgerRepository;
use Modules\Amazon\Repositories\AmazonLedgerSummaryRepository;
use Spatie\LaravelData\DataCollection;

class AmazonManager
{
    private AmazonLedgerRepository $ledgers;
    private AmazonLedgerSummaryRepository $summaries;

    public function __construct(private readonly AmazonIntegrationInstance $integrationInstance)
    {
        $this->ledgers = app(AmazonLedgerRepository::class);
        $this->summaries = app(AmazonLedgerSummaryRepository::class);
    }

    /**
     * @throws Exception
     */
    public function fbaInventoryValuation(?Carbon $date = null): FbaInventoryValuationData
    {
        if (!$date) {
            $date = Carbon::parse(request()->input('filter')['before_event_date'])->addSecond()->subDay();
        }

        $fnskuProductsData = $this->getValuationFromInitialInventory();
        $this->addValuationFromLedgers($fnskuProductsData);
        $this->addValuationForAdjustmentsFromSummaries($fnskuProductsData);
        $this->addSummaryQuantities($fnskuProductsData, $date);
        $this->fnskuProductsDataCalculations($fnskuProductsData);

        return FbaInventoryValuationData::from([
            'totalQuantity' => $fnskuProductsData->toCollection()->sum('quantity'),
            'totalValuation' => $fnskuProductsData->toCollection()->sum('valuation'),
            'totalSummaryQuantity' => $fnskuProductsData->toCollection()->sum('summaryQuantity'),
            'ledgerReconciliationDate' => (new AmazonLedgerManager($this->integrationInstance))->getLastReconciledDate()->toDateString(),
            'summaryReconciliationDate' => (new AmazonLedgerSummaryManager($this->integrationInstance))->getLastReconciledDate()->toDateString(),
            'products' => $fnskuProductsData,
        ]);
    }

    private function getValuationFromInitialInventory(): DataCollection
    {
        return FbaInventoryValuationFnskuProductData::collection(
            $this->ledgers->getReconciledFromInitialInventoryReport($this->integrationInstance)->map(function (AmazonFbaInitialInventory $initialInventory) {
                $totalQuantity = $initialInventory->ending_warehouse_balance + $initialInventory->in_transit_between_warehouses;
                return FbaInventoryValuationFnskuProductData::from([
                    'fnskuProduct' => $initialInventory->amazonFnskuProduct,
                    'quantity' => $totalQuantity,
                    'quantityFromInitialInventory' => $totalQuantity,
                    'valuation' => $initialInventory->fifoLayer->avg_cost * $totalQuantity,
                ]);
            })
        );
    }

    private function addValuationForAdjustmentsFromSummaries(DataCollection $fnskuProductsData): void
    {
        $this->summaries->getSummariesWithAdjustmentsForValuation($this->integrationInstance)->each(function (AmazonFbaReportInventoryLedgerSummary $summary) use ($fnskuProductsData)
        {
            $movements = $summary->inventoryAdjustments->flatMap(function (InventoryEvent $skuLink) {
                return $skuLink->inventoryMovements;
            });
            $cost = $movements->map(function (InventoryMovement $movement) {
                return $movement->fifo_layer->avg_cost * $movement->quantity;
            })->sum();

            $quantity = $summary->inventoryAdjustments->sum('quantity');
            if ($fbaInventoryValuationData = $fnskuProductsData
                ->where('fnskuProduct.id', $summary->amazonFnskuProduct->id)
                ->first())
            {
                $fbaInventoryValuationData->quantity += $quantity;
                customlog('amazon', $quantity . ' from Ledger Summary ID: ' . $summary->id . ' date: ' . $summary->date);
                $fbaInventoryValuationData->quantityFromSummary += $quantity;
                $fbaInventoryValuationData->valuation += $cost;
            } else {
                $fnskuProductsData[] = FbaInventoryValuationFnskuProductData::from([
                    'fnskuProduct' => $summary->amazonFnskuProduct,
                    'quantity' => $quantity,
                    'quantityFromSummary' => $quantity,
                    'valuation' => $cost,
                ]);
            }
        });
    }

    private function addValuationFromLedgers(DataCollection $fnskuProductsData): void
    {
        $this->ledgers->getLedgersForValuation($this->integrationInstance)->each(function (AmazonLedgerForValuationData $amazonLedgerForValuationData) use ($fnskuProductsData)
        {
            $cost = $amazonLedgerForValuationData->inventoryMovements
                // Reject where the movement doesn't belong to the fifo layer like with warehouse transfers
                ->reject(fn (InventoryMovement $movement) => $movement->warehouse_id != $movement->fifo_layer->warehouse_id)
                ->map(function (InventoryMovement $movement) {
                    return $movement->fifo_layer->avg_cost * $movement->quantity;
                })->sum();

            if ($fbaInventoryValuationData = $fnskuProductsData
                ->where('fnskuProduct.id', $amazonLedgerForValuationData->fnskuProduct->id)
                ->first())
            {
                $fbaInventoryValuationData->quantity += $amazonLedgerForValuationData->quantity;
                $fbaInventoryValuationData->quantityFromLedgers += $amazonLedgerForValuationData->quantity;
                $fbaInventoryValuationData->valuation += $cost;
            } else {
                $fnskuProductsData[] = FbaInventoryValuationFnskuProductData::from([
                    'fnskuProduct' => $amazonLedgerForValuationData->fnskuProduct,
                    'quantity' => $amazonLedgerForValuationData->quantity,
                    'quantityFromLedgers' => $amazonLedgerForValuationData->quantity,
                    'valuation' => $cost,
                ]);
            }
        });
    }

    private function addSummaryQuantities(DataCollection $fnskuProductsData, Carbon $date): void
    {
        $this->summaries->getSummariesForValuationDate($this->integrationInstance, $date)->each(function (AmazonFbaReportInventoryLedgerSummary $summary) use ($fnskuProductsData)
        {
            if ($fbaInventoryValuationData = $fnskuProductsData
                ->where('fnskuProduct.id', $summary->amazonFnskuProduct->id)
                ->first())
            {
                $fbaInventoryValuationData->summaryQuantity += $summary->total_inventory_quantity;
            }
            else {
                $fnskuProductsData[] = FbaInventoryValuationFnskuProductData::from([
                    'fnskuProduct' => $summary->amazonFnskuProduct,
                    'quantity' => 0,
                    'valuation' => 0,
                    'summaryQuantity' => $summary->total_inventory_quantity,
                ]);
            }
        });
    }

    private function fnskuProductsDataCalculations(DataCollection $fnskuProductsData): void
    {
        $fnskuProductsData = $fnskuProductsData->map(function (FbaInventoryValuationFnskuProductData $fnskuProductData) {
            $fnskuProductData->averageCost = $fnskuProductData->quantity == 0 ? 0 : round($fnskuProductData->valuation / $fnskuProductData->quantity, 2);
            $fnskuProductData->discrepancy = $fnskuProductData->quantity - $fnskuProductData->summaryQuantity;
            $fnskuProductData->valuation = round($fnskuProductData->valuation, 2);
            return $fnskuProductData;
        });
    }


    // TODO: Review with Jatin to see if we can accomplish all in mysql
//    public function fbaInventoryValuation(AmazonIntegrationInstance $integrationInstance): EloquentCollection
//    {
//        $query = QueryBuilder::for(AmazonFbaReportInventoryLedger::class)
//            ->where('integration_instance_id', $integrationInstance->id)
//            ->allowedFilters([
//                AllowedFilter::scope('before_event_date')
//            ])
//            ->leftJoin('inventory_adjustments', function(JoinClause $join) {
//                $join->on('inventory_adjustments.id', '=', 'amazon_fba_report_inventory_ledger.sku_link_id');
//                $join->where('amazon_fba_report_inventory_ledger.sku_link_type', InventoryAdjustment::class);
//            })
//            ->leftJoin('sales_credit_return_lines', function(JoinClause $join) {
//                $join->on('sales_credit_return_lines.id', '=', 'amazon_fba_report_inventory_ledger.sku_link_id');
//                $join->where('amazon_fba_report_inventory_ledger.sku_link_type', SalesCreditReturnLine::class);
//            })
//            ->join('amazon_fnsku_products', function (JoinClause $join) {
//                $join->on('amazon_fba_report_inventory_ledger.integration_instance_id', '=', 'amazon_fnsku_products.integration_instance_id');
//                $join->on('amazon_fba_report_inventory_ledger.fnsku', '=', 'amazon_fnsku_products.fnsku');
//                $join->on('amazon_fba_report_inventory_ledger.country', '=', 'amazon_fnsku_products.location');
//                $join->on('amazon_fba_report_inventory_ledger.disposition', '=', 'amazon_fnsku_products.disposition');
//            })
//            ->groupBy('fnsku.id');
//
//        return $query->get();
//    }
}
