<?php

namespace Modules\Amazon\Managers;

use App\DTO\InventoryAdjustmentDto;
use App\Exceptions\PurchaseOrder\NotOpenPurchaseOrderException;
use App\Exceptions\WarehouseTransfers\NotOpenWarehouseTransferException;
use App\Models\Product;
use App\Repositories\InventoryAdjustmentRepository;
use App\Services\InventoryManagement\InventoryManager;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Pipeline;
use Modules\Amazon\Actions\LedgerActions\AmazonCheckForUnreconciledLedgers;
use Modules\Amazon\Actions\LedgerActions\AmazonCheckIfDetailReportWasProcessed;
use Modules\Amazon\Actions\LedgerActions\AmazonCheckIfLedgerIsReconciled;
use Modules\Amazon\Actions\LedgerActions\AmazonProcessSkuModel;
use Modules\Amazon\Data\AmazonFbaLedgerReconciliationReportData;
use Modules\Amazon\Data\AmazonFbaLedgerReconciliationReportProductData;
use Modules\Amazon\Data\AmazonReportInfoData;
use Modules\Amazon\Entities\AbstractAmazonFbaDetailReport;
use Modules\Amazon\Entities\AmazonFbaInitialInventory;
use Modules\Amazon\Entities\AmazonFbaReportInventoryLedger;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Enums\Entities\FbaInventoryLedgerReportEventTypeEnum;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Exceptions\LedgerEventTypeWithNoDetailReportException;
use Modules\Amazon\Exceptions\LedgerUnreconciledOutOfOrderException;
use Modules\Amazon\Repositories\AmazonFbaInboundShipmentRepository;
use Modules\Amazon\Repositories\AmazonFnskuRepository;
use Modules\Amazon\Repositories\AmazonLedgerRepository;
use Modules\Amazon\Repositories\AmazonOrderItemRepository;
use Modules\Amazon\Repositories\AmazonOrderRepository;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Throwable;

class AmazonLedgerManager
{
    private AmazonLedgerRepository $ledgers;
    private AmazonFnskuRepository $fnskus;
    private InventoryAdjustmentRepository $adjustments;
    private AmazonReportRepository $reports;

    public function __construct(private readonly AmazonIntegrationInstance $integrationInstance)
    {
        $this->ledgers = app(AmazonLedgerRepository::class);
        $this->fnskus = app(AmazonFnskuRepository::class);
        $this->adjustments = app(InventoryAdjustmentRepository::class);
        $this->reports = app(AmazonReportRepository::class);
    }

    /**
     *  The reconcile ledger process is the process of matching Amazon FBA ledger events with SKU events so that:
     *  1. We can maintain our own audit trail on the FBA warehouse
     *  2. We can keep track of FIFO so that COGS is accurate
     *
     * @throws Throwable
     * @throws NotOpenPurchaseOrderException
     * @throws NotOpenWarehouseTransferException
     */
    public function reconcileLedgers(?AmazonReport $amazonReport = null, ?AmazonFnskuProduct $fnsku = null, ?array $ids = null): Collection
    {
        $this->markReconciliationErrorsForProblemFnskus($amazonReport);

        $fnskuProducts = $fnsku ?
            collect()->add($fnsku) :
            $this->fnskus->sanitizeFnskuProductsForLedgerReconciliation($this->integrationInstance, $amazonReport, $ids);

        $ledgerCollection = new Collection();

        $fnskuProducts->each(function (AmazonFnskuProduct $fnskuProduct) use ($amazonReport, $ledgerCollection, $ids)
        {
            customlog('amazon', "\tReconciling ledgers for FNSKU ID: $fnskuProduct->id, $fnskuProduct->fnsku $fnskuProduct->location, $fnskuProduct->disposition)...");
            customlog('amazon', "\tStarting tallies reconciled: $fnskuProduct->reconciled_quantity");

            $unreconciledLedgersGroupedByDate = $this->ledgers->getLedgerFnskuProductGroupedByDate($fnskuProduct, $ids);

            foreach ($unreconciledLedgersGroupedByDate as $event_date => $ledgers)
            {
                customlog('amazon', "Processing for date: $event_date");

                /*
                 * TODO: Consider breaking up reconcileForFnskuProduct and corresponding getUnreconciledForFnskuProduct into two methods
                 *  that make sense to run before and after ledger processing (and named appropriately(
                 */

                // Process summary before processing ledgers, which will only process summaries up to the day before the ledger date
                if ($summary = $ledgers->first()->ledgerSummary) {
                    customlog('amazon', "Reconciling summaries");
                    (new AmazonLedgerSummaryManager($this->integrationInstance))->reconcileForFnskuProduct($summary);
                }

                $sortedLedgers = $this->ledgers->sortLedgersForProcessing($fnskuProduct->refresh(), $ledgers);
                foreach ($sortedLedgers as $ledger)
                {
                    $processedLedger = $this->reconcile($ledger);
                    $ledgerCollection->push($processedLedger);
                    if (!is_null($processedLedger->errorLog)) {
                        $ledgerCollection = $ledgerCollection->merge($this->ledgers->markReconciliationErrorsForHaltedBy($processedLedger));
                        // Skip to next fnskuProducts iteration
                        break;
                    }
                }

                // Skip to next fnskuProducts iteration if the ledgers for the day are not completely reconciled
                if ($ledgers->whereNull('reconciled_at')->isNotEmpty()) {
                    break;
                }

                // Process summary after processing ledgers, which will process the day of the ledger and after for any days when there are no ledgers
                if ($summary = $ledgers->first()->ledgerSummary) {
                    customlog('amazon', "Reconciling summaries");
                    (new AmazonLedgerSummaryManager($this->integrationInstance))->reconcileForFnskuProduct($summary);
                }
            }
        });
        return $ledgerCollection;
    }

    /**
     * @throws Throwable
     */
    public function reconcile(AmazonFbaReportInventoryLedger $ledger): AmazonFbaReportInventoryLedger
    {
        customlog('amazon', "Reconciling ledger for ID: $ledger->id, $ledger->event_datetime, " . $ledger->event_type->value . ", $ledger->quantity, $ledger->fnsku, $ledger->disposition...");
        return DB::transaction(function () use ($ledger) {
            return Pipeline::send($ledger)
                ->through([
                    AmazonCheckIfLedgerIsReconciled::class,
                    AmazonCheckForUnreconciledLedgers::class,
                    AmazonCheckIfDetailReportWasProcessed::class,
                    AmazonProcessSkuModel::class,
                ])
                ->thenReturn();
        });
    }

    private function markReconciliationErrorsForProblemFnskus(?AmazonReport $amazonReport): void
    {
        $this->ledgers->markReconciliationErrorsForMissingInitialInventoryReport($this->integrationInstance, $amazonReport);
        $this->ledgers->markReconciliationErrorsForMissingAmazonFnskus($this->integrationInstance, $amazonReport);
        $this->ledgers->markReconciliationErrorsForUninitializedAmazonFnskus($this->integrationInstance, $amazonReport);
    }

    /**
     * @throws LedgerEventTypeWithNoDetailReportException
     */
    public function linkLedgerToDetailReports(AmazonFbaReportInventoryLedger $ledger, array $detailReportIds): AmazonFbaReportInventoryLedger
    {
        $detailReports = [];
        foreach ($detailReportIds as $detailReportId) {
            // Get detail model for ledger
            $detailReportClass = $this->ledgers->getDetailModelFromLedgerEventType($ledger->event_type);

            $detailReports[] = $detailReportClass::with('ledger')->findOrFail($detailReportId);
        }

        $this->ledgers->linkDetailsToLedger($ledger, $detailReports);
        $ledger->errorLog = null;
        $ledger->update();

        return $ledger->load('ledgerDetails.detail')->refresh();
    }

    /**
     * This links Amazon FBA Ledger Reports to Detail Reports to support createAuditTrailFromAmazonLedger()
     *
     * @throws Exception
     */
    public function linkAllLedgersToDetailReports(): void
    {
        customlog('amazon', 'Linking Amazon FBA Ledger Reports to Detail Reports');
        $unlinkedLedgerReports = $this->ledgers->getLedgerReportsNotLinkedToDetail();

        $numUnlinkedLedgerReports = $unlinkedLedgerReports->count();
        customlog('amazon', $numUnlinkedLedgerReports.' Unlinked Amazon FBA Ledger Reports to Link');

        $unlinkedLedgerReports->each(
        /**
         * @throws Exception
         */
            function (AmazonFbaReportInventoryLedger $unlinkedLedgerReport) use (&$numUnlinkedLedgerReports) {
                $numUnlinkedLedgerReports--;
                customlog('amazon', 'Linking Amazon FBA Ledger Report: '.$unlinkedLedgerReport->id.', '.$numUnlinkedLedgerReports.' remaining');

                $detailMatches = $this->ledgers->getDetailReportLedgerMatches($unlinkedLedgerReport);
                //customlog("amazon", "\tFound ".$detailMatches->count()." matches");

                if ($detailMatches->count() == 1) {
                    /** @var AbstractAmazonFbaDetailReport $detailReport */
                    $detailReport = $detailMatches->first();
                    $this->ledgers->linkDetailsToLedger($unlinkedLedgerReport, [$detailReport]);
                    $unlinkedLedgerReport->errorLog = null;
                    $unlinkedLedgerReport->update();
                }
                else {

                    // More than one match for a strict date... see if quantity adds up to ledger quantity, if so, match  on all
                    $detailMatches = $this->ledgers->getDetailReportLedgerMatches($unlinkedLedgerReport, 1, true);

                    if ($detailMatches->count() >= 2 && abs($unlinkedLedgerReport->quantity) == $detailMatches->sum('detailQuantity')) {
                        $this->ledgers->linkDetailsToLedger($unlinkedLedgerReport, $detailMatches->all());
                        $unlinkedLedgerReport->errorLog = null;
                        $unlinkedLedgerReport->update();
                    } // Try again but with a more flexible date match
                    else {
                        $detailMatches = $this->ledgers->getDetailReportLedgerMatches($unlinkedLedgerReport, 1);
                        //customlog("amazon", "\t\tFound ".$detailMatches->count()." matches on retry");
                        if ($detailMatches->count() == 1) {
                            /** @var AbstractAmazonFbaDetailReport $detailReport */
                            $detailReport = $detailMatches->first();
                            $this->ledgers->linkDetailsToLedger($unlinkedLedgerReport, [$detailReport]);
                            $unlinkedLedgerReport->errorLog = null;
                            $unlinkedLedgerReport->update();
                        }
                        else
                        {
                            $detailMatches = $this->ledgers->getDetailReportLedgerMatches($unlinkedLedgerReport, 1);
                            if ($detailMatches->count() >= 2 && abs($unlinkedLedgerReport->quantity) == $detailMatches->sum('detailQuantity')) {
                                $this->ledgers->linkDetailsToLedger($unlinkedLedgerReport, $detailMatches->all());
                                $unlinkedLedgerReport->errorLog = null;
                                $unlinkedLedgerReport->update();
                            }
                            elseif ($detailMatches->first()?->detailQuantity == abs($unlinkedLedgerReport->quantity)) {
                                $this->ledgers->linkDetailsToLedger($unlinkedLedgerReport, [$detailMatches->first()]);
                                $unlinkedLedgerReport->errorLog = null;
                                $unlinkedLedgerReport->update();
                            }
                            else {
                                $detailMatches = $this->ledgers->getDetailReportLedgerMatches($unlinkedLedgerReport, 1, false, true);
                                if ($detailMatches->count() == 1) {
                                    /** @var AbstractAmazonFbaDetailReport $detailReport */
                                    $detailReport = $detailMatches->first();
                                    $this->ledgers->linkDetailsToLedger($unlinkedLedgerReport, [$detailReport]);
                                    $unlinkedLedgerReport->errorLog = null;
                                    $unlinkedLedgerReport->update();
                                }
                            }
                        }
                    }
                }
            });
        customlog('amazon', 'Finished Linking Amazon FBA Ledger Reports to Detail Reports.  IDs: ' . $unlinkedLedgerReports->pluck('id')->implode(','));
    }

    public function unreconcileLedgers(?AmazonFnskuProduct $fnsku = null, ?array $ids = null): Collection
    {
        $fnskuProducts = $fnsku ?
            collect()->add($fnsku) :
            $this->fnskus->sanitizeFnskuProductsForLedgerUnreconciliation($this->integrationInstance, $ids);

        $ledgerCollection = collect();

        $fnskuProducts->each(/**
         * @throws Throwable
         */ function (AmazonFnskuProduct $fnskuProduct) use ($ledgerCollection, $ids)
        {
            customlog('amazon', "Unreconciling ledgers for ID: $fnskuProduct->id, $fnskuProduct->fnsku $fnskuProduct->location, $fnskuProduct->disposition)...");
            customlog('amazon', "\tEnding tallies reconciled: $fnskuProduct->reconciled_quantity");

            $reconciledLedgersGroupedByDate = $this->ledgers->getLedgerFnskuProductGroupedByDate($fnskuProduct, $ids, false);

            foreach ($reconciledLedgersGroupedByDate as $event_date => $ledgers)
            {
                customlog('amazon', "Processing for date: $event_date");
                // Process summary before processing ledgers, which will only process summaries after the ledger date
                if ($summary = $ledgers->first()->ledgerSummary) {
                    (new AmazonLedgerSummaryManager($this->integrationInstance))->unreconcileForFnskuProduct($summary);
                }
                $sortedLedgers = $this->ledgers->sortLedgersForProcessing($fnskuProduct->refresh(), $ledgers, false);
                $sortedLedgers->each(function (AmazonFbaReportInventoryLedger $ledger) use ($ledgerCollection) {
                    $ledgerCollection->push($this->unreconcileLedger($ledger));
                });
                // Process summary after processing ledgers, which will process the day of the ledger and before for any days when there are no ledgers
                if ($summary = $ledgers->first()->ledgerSummary) {
                    (new AmazonLedgerSummaryManager($this->integrationInstance))->unreconcileForFnskuProduct($summary);
                }
            }
        });

        return $ledgerCollection->sort(function ($a, $b) {
                // Compare the event_datetime first
                if ($a->event_datetime === $b->event_datetime) {
                    // event_datetime is the same, sort by cast_quantity descending
                    if ($a->quantity === $b->quantity) {
                        return 0; // the items are equivalent
                    }
                    return $a->quantity < $b->quantity ? 1 : -1;
                }
                return $a->event_datetime > $b->event_datetime ? 1 : -1;
            })->values();
    }

    /**
     * @throws Throwable
     */
    public function unreconcileLedger(AmazonFbaReportInventoryLedger $amazonFbaReportInventoryLedger): AmazonFbaReportInventoryLedger
    {
        $reconciledLedgersCount = app(AmazonFnskuRepository::class)->countReconciledLedgersAfterLedger($amazonFbaReportInventoryLedger);
        if ($reconciledLedgersCount > 0) {
            throw new LedgerUnreconciledOutOfOrderException("Ledger $amazonFbaReportInventoryLedger->id unreconciled out of order. There are $reconciledLedgersCount reconciled ledgers after this one.");
        }
        DB::transaction(function () use ($amazonFbaReportInventoryLedger) {
            // Should not delete the sku link if it is not a multi-channel fulfillment
            if ($amazonFbaReportInventoryLedger->shipments->count() == 0 || $amazonFbaReportInventoryLedger->shipments->first()->sales_channel != 'Non-Amazon') {
                $amazonFbaReportInventoryLedger->skuLink?->delete();
            }
            $this->clearLedgerReconciliation($amazonFbaReportInventoryLedger);
            $amazonFbaReportInventoryLedger->amazonFnskuProduct->reconciled_quantity -= $amazonFbaReportInventoryLedger->quantity;
            $amazonFbaReportInventoryLedger->amazonFnskuProduct->save();
        });
        return $amazonFbaReportInventoryLedger->refresh();
    }

    public function clearLedgerReconciliation(AmazonFbaReportInventoryLedger $ledger): void
    {
        $this->ledgers->clearLedgerReconciliation($ledger);
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function createInventoryAdjustmentFromLedger(
        AmazonFbaReportInventoryLedger $amazonFbaReportInventoryLedger,
        string $notes,
        ?int $qtyOverride = null,
    ): AmazonFbaReportInventoryLedger
    {
        customlog('amazon', 'Creating inventory adjustment from ledger');

        $product = $amazonFbaReportInventoryLedger->amazonFnskuProduct->product;
        $quantity = $qtyOverride ?? $amazonFbaReportInventoryLedger->quantity;

        $data = InventoryAdjustmentDto::from([
            'adjustment_date' => Carbon::parse($amazonFbaReportInventoryLedger->event_datetime),
            'product_id' => $product->id,
            'warehouse_id' => $this->integrationInstance->warehouse->id,
            'quantity' => $quantity,
            'notes' => $notes,
        ]);

        $inventoryAdjustment = $this->adjustments->createInventoryAdjustment($data->toArray());

        $inventoryManager = InventoryManager::with(
            $inventoryAdjustment->warehouse_id,
            $inventoryAdjustment->product
        );
        if ($inventoryAdjustment->quantity > 0) {
            $inventoryManager->addToStock(abs($inventoryAdjustment->quantity), $inventoryAdjustment);
        } elseif ($inventoryAdjustment->quantity < 0) {
            $inventoryManager->takeFromStock(abs($inventoryAdjustment->quantity), $inventoryAdjustment);
        } else {
            throw new Exception('Inventory Adjustment quantity is 0 for amazon fba ledger ' . $amazonFbaReportInventoryLedger->id);
        }

        customlog('amazon', 'Clearing audit trail error');
        $amazonFbaReportInventoryLedger->skuLink()->associate($inventoryAdjustment);
        $amazonFbaReportInventoryLedger->last_reconciliation_attempted_at = Carbon::now();
        $amazonFbaReportInventoryLedger->reconciled_at = Carbon::now();
        $amazonFbaReportInventoryLedger->errorLog = null;
        $amazonFbaReportInventoryLedger->blocked_by_ledger_id = null;
        $amazonFbaReportInventoryLedger->save();
        $amazonFbaReportInventoryLedger->amazonFnskuProduct->reconciled_quantity += $amazonFbaReportInventoryLedger->quantity;
        $amazonFbaReportInventoryLedger->amazonFnskuProduct->save();

        return $amazonFbaReportInventoryLedger;
        //Log::debug("\tDetail report was processed for $amazonFbaReportInventoryLedger->event_datetime with no match found for ledger, so creating adjustment instead.");
    }

    public function clearAuditTrailFromAmazonLedger(Product $product): void
    {
        $amazonFnskuProducts = $this->fnskus->getAmazonFnskuProductsFromProduct($product);

        $amazonFnskuProducts->each(function (AmazonFnskuProduct $amazonFnskuProduct) {
            $amazonFnskuProduct->amazonFbaReportInventoryLedgers
                ->sort(function ($a, $b) {
                    return [$b->event_datetime, $a->quantity] <=> [$a->event_datetime, $b->quantity];
                })
                ->each(function (AmazonFbaReportInventoryLedger $amazonFbaReportInventoryLedger) {
                    // Delete link model
                    $this->ledgers->deleteSkuModelFromLedger($amazonFbaReportInventoryLedger);
                });
            // Delete initial count
            $amazonFnskuProduct->amazonFbaInitialInventory->each(function (AmazonFbaInitialInventory $amazonFbaInitialInventory) {
                if ($fifoLayer = $amazonFbaInitialInventory->fifoLayer) {
                    $amazonFbaInitialInventory->sku_product_initialized_at = null;
                    $amazonFbaInitialInventory->fifo_layer_id              = null;
                    $amazonFbaInitialInventory->save();
                    $fifoLayer->delete();
                }
            });
        });
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function resetDetailReport(AmazonReport $amazonReport): void
    {
        $model = $this->reports->getModelClassFromReportType($amazonReport->reportType);

        DB::transaction(function () use ($model, $amazonReport) {
            $model::where('amazon_report_id', $amazonReport->id)->each(function (AbstractAmazonFbaDetailReport $detailReport) {
                if ($ledger = $detailReport->ledger) {
                    $this->ledgers->deleteSkuModelFromLedger($ledger);
                    $this->ledgers->clearLedgerDetails($ledger);
                }
                $detailReport->delete();
            });
        });

        $amazonReport->processed_at = null;
        $amazonReport->save();
    }

    /**
     * @throws Exception
     */
    public function getLastReconciledDate(): Carbon
    {
        if ($oldestUnreconciledDate = $this->ledgers->getOldestUnreconciledLedgerDate($this->integrationInstance))
        {
            return Carbon::parse($oldestUnreconciledDate, $this->integrationInstance->getTimezone());
        }

        if ($lastReconciledDate = $this->ledgers->getLastReconciledLedgerDate($this->integrationInstance)) {
            return Carbon::parse($lastReconciledDate, $this->integrationInstance->getTimezone());
        }

        return $this->integrationInstance->fbaInventoryTrackingStartDate()->setTimezone($this->integrationInstance->getTimezone());
    }

    /**
     * @throws Exception
     */
    public function reconciliationReport(): AmazonFbaLedgerReconciliationReportData
    {
        $fbaInventoryStartDate = $this->integrationInstance->fbaInventoryTrackingStartDate()->setTimezone($this->integrationInstance->getTimezone());
        $currentDate = Carbon::now($this->integrationInstance->getTimezone());
        $daysSinceStartCurrent = $currentDate->diffInDays($fbaInventoryStartDate);

        $lastReconciledDate = $this->getLastReconciledDate();
        $reconciledDaysSinceStart = $lastReconciledDate?->diffInDays($fbaInventoryStartDate);

        $fnskuReconciliationBreakdown = $this->ledgers
            ->getReconciliationDatesByFnsku($this->integrationInstance)
            ->map(function ($item) use ($fbaInventoryStartDate) {
                $maxDate = $item->max_date ? Carbon::parse($item->max_date, $this->integrationInstance->getTimezone()) : null;
                $maxReconciledDate = $item->max_reconciled_date ? Carbon::parse($item->max_reconciled_date, $this->integrationInstance->getTimezone()) : null;

                $daysSinceStart = $maxDate ? $maxDate->diffInDays($fbaInventoryStartDate) : 0;
                $daysSinceStartReconciled = $maxReconciledDate ? $maxReconciledDate->diffInDays($fbaInventoryStartDate) : 0;

                return [
                    'fnsku' => $item->fnsku,
                    'fnskuProductId' => $item->fnskuProductId,
                    'msku' => $item->msku,
                    'sku' => $item->sku,
                    'daysSinceStartReconciled' => $daysSinceStartReconciled,
                    'daysSinceStartLedgerData' => $daysSinceStart,
                    'reconciliationDate' => $maxReconciledDate?->toDateString(),
                    'latestLedgerDate' => $maxDate?->toDateString(),
                ];
            });

        return AmazonFbaLedgerReconciliationReportData::from([
            'products' => AmazonFbaLedgerReconciliationReportProductData::collection($fnskuReconciliationBreakdown),
            'startDate' => $fbaInventoryStartDate->addSecond()->toDateString(), // TODO: Should fba inventory tracking start date be stored in integration instances using the 59:59?
            'currentDate' => $currentDate->toDateString(),
            'daysSinceStartCurrent' => $daysSinceStartCurrent,
            'reconciliationDate' => $lastReconciledDate->toDateString(),
            'reconciledDaysSinceStart' => $reconciledDaysSinceStart,
            'ledgersReconciled' => $this->ledgers->getLedgersReconciledCount($this->integrationInstance),
            'ledgersUnreconciled' => $this->ledgers->getLedgersReconciledCount($this->integrationInstance, false),
            'topErrors' => $this->ledgers->getTopLedgerErrors($this->integrationInstance)->toArray(),
        ]);
    }

    /**
     * @throws Exception
     */
    public function getInfo(AmazonReportTypeEnum $reportTypeEnum): AmazonReportInfoData
    {
        $lastRequest = $this->reports->getLastRequestedForType($reportTypeEnum, $this->integrationInstance);
        $lastReportInProcess = $this->reports->getLastReportInProcessForType($reportTypeEnum, $this->integrationInstance);
        $lastReport = $this->reports->getLastReportForType($reportTypeEnum, $this->integrationInstance);

//        $nextSchedule = app(ScheduleManager::class)->commandNextSchedule(
//            new RequestAmazonReportCommand(),
//            $this->integrationInstance->id . ' ' . $reportTypeEnum->value
//        );

        return AmazonReportInfoData::from([
            'reportType' => $reportTypeEnum,
            'reportRequest' => $lastRequest,
            'reportInProcess' => $lastReportInProcess,
            'report' => $lastReport,
        ]);
    }
}