<?php

namespace Modules\Amazon\Managers;

use App\Abstractions\Integrations\ClientResponseDataInterface;
use App\Abstractions\Integrations\IntegrationClientInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelManager;
use App\Exceptions\ApiException;
use Exception;
use Illuminate\Http\Client\ConnectionException;
use Illuminate\Support\Facades\Artisan;
use Modules\Amazon\Actions\InitializeFbaShippingMethods;
use Modules\Amazon\Actions\InitializeFbaWarehouse;
use Modules\Amazon\Actions\RequestAmazonReport;
use Modules\Amazon\Data\RequestAmazonReportData;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Enums\Entities\AmazonReportTypeEnum;
use Modules\Amazon\Exceptions\AmazonTimeoutException;
use Modules\Amazon\Jobs\RefreshAmazonFbaInboundShipmentsJob;
use Modules\Amazon\Jobs\RefreshAmazonSettlementReportsJob;
use Modules\Amazon\Repositories\AmazonIntegrationInstanceRepository;
use Modules\Amazon\Services\AmazonClient;
use Throwable;

class AmazonIntegrationInstanceManager extends AbstractSalesChannelManager
{
    protected IntegrationClientInterface $client;
    protected AmazonIntegrationInstanceRepository $instances;

    /**
     * @throws Exception
     */
    public function __construct(protected AmazonIntegrationInstance $amazonIntegrationInstance)
    {
        $this->client = new AmazonClient($this->amazonIntegrationInstance);
        $this->instances = app(AmazonIntegrationInstanceRepository::class);
        parent::__construct($amazonIntegrationInstance, new AmazonClient($amazonIntegrationInstance));
    }

    /**
     * @throws Throwable
     */
    public function processAmazonFbaStateChange(): void
    {
        customlog('amazon', 'Processing FBA State Change');
        $integrationSettings = $this->amazonIntegrationInstance->integration_settings;
        if ($integrationSettings['is_fba_enabled']) {
            (new InitializeFbaWarehouse($this->amazonIntegrationInstance))->handle();
            (new InitializeFbaShippingMethods($this->amazonIntegrationInstance))->handle();

            (new AmazonInitialInventoryManager($this->amazonIntegrationInstance))->requestInitialInventory();
            (new RequestAmazonReport(RequestAmazonReportData::from([
                'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
                'report_type' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER,
                'createImmediately' => false,
            ])))->handle();
            (new RequestAmazonReport(RequestAmazonReportData::from([
                'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
                'report_type' => AmazonReportTypeEnum::FBA_REPORT_INVENTORY_LEDGER_SUMMARY,
                'createImmediately' => false,
            ])))->handle();
            (new RequestAmazonReport(RequestAmazonReportData::from([
                'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
                'report_type' => AmazonReportTypeEnum::FBA_INVENTORY,
                'createImmediately' => false,
            ])))->handle();
            // FBA_REPORT_SHIPMENTS is a splittable report handled within the RequestAmazonReport
            (new RequestAmazonReport(RequestAmazonReportData::from([
                'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
                'report_type' => AmazonReportTypeEnum::FBA_REPORT_SHIPMENTS,
                'data_start_date' => $this->amazonIntegrationInstance->getLedgerStartDate()->subDays(7),
                'createImmediately' => false,
            ])))->handle();
            (new RequestAmazonReport(RequestAmazonReportData::from([
                'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
                'report_type' => AmazonReportTypeEnum::FBA_REPORT_CUSTOMER_RETURNS,
                'data_start_date' => $this->amazonIntegrationInstance->getLedgerStartDate()->subDays(7),
                'createImmediately' => false,
            ])))->handle();
            (new RequestAmazonReport(RequestAmazonReportData::from([
                'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
                'report_type' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_ORDERS,
                'data_start_date' => $this->amazonIntegrationInstance->getLedgerStartDate()->subDays(60),
                'createImmediately' => false,
            ])))->handle();
            (new RequestAmazonReport(RequestAmazonReportData::from([
                'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
                'report_type' => AmazonReportTypeEnum::FBA_REPORT_REMOVAL_SHIPMENTS,
                'data_start_date' => $this->amazonIntegrationInstance->getLedgerStartDate()->subDays(60),
                'createImmediately' => false,
            ])))->handle();
            #Artisan::call('amazon:reports:create');

            dispatch(new RefreshAmazonFbaInboundShipmentsJob($this->amazonIntegrationInstance));
        } else {
            // If FBA is disabled, we de-initialize the FBA Warehouse
            (new InitializeFbaWarehouse($this->amazonIntegrationInstance, false))->handle();
        }
    }

    /**
     * @throws Throwable
     */
    public function requestInitialReports(): void
    {
        customlog('amazon', 'Requesting initial reports for Amazon Integration Instance: '.$this->amazonIntegrationInstance->name);
        (new RequestAmazonReport(RequestAmazonReportData::from([
            'amazonIntegrationInstance' => $this->amazonIntegrationInstance,
            'report_type' => AmazonReportTypeEnum::PRODUCTS,
            'createImmediately' => false,
        ])))->handle();
        dispatch(new RefreshAmazonSettlementReportsJob($this->amazonIntegrationInstance));
        Artisan::call('amazon:reports:create');
    }

    /**
     * @throws AmazonTimeoutException
     * @throws ApiException
     * @throws ConnectionException
     */
    public function getMarketplaceParticipations(): ClientResponseDataInterface
    {
        $responseDto = $this->client->getMarketplaceParticipations();

        if (! $responseDto->collection->count()) {
            return $responseDto;
        }

        $this->instances->updateIntegrationSettings($this->integrationInstance, [
            'marketplaceParticipations' => $responseDto->collection->toArray(),
            'catalogMarketplaceId' => $this->getCatalogMarketplaceId()
        ]);

        return $responseDto;
    }

    public function getCatalogMarketplaceId(): string
    {
        return match ($this->amazonIntegrationInstance->country) {
            'EU' => 'A1PA6795UKMFR9', // UK
            default => 'ATVPDKIKX0DER', // US
        };
    }
}
