<?php

namespace Modules\Amazon\Managers;

use App\Abstractions\AbstractRepository;
use App\Abstractions\Integrations\ClientResponseDataInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelManager;
use App\Data\IdQuantityData;
use App\DTO\PurchaseOrderDto;
use App\DTO\PurchaseOrderLineDto;
use App\DTO\WarehouseTransferDto;
use App\DTO\WarehouseTransferLineDto;
use App\Exceptions\SupplierProductNotFoundException;
use App\Helpers;
use App\Managers\WarehouseTransferManager;
use App\Models\Currency;
use App\Models\Product;
use App\Models\PurchaseOrder;
use App\Models\Setting;
use App\Models\Supplier;
use App\Models\SupplierProduct;
use App\Models\Warehouse;
use App\Models\WarehouseTransfer;
use App\Repositories\PurchaseOrderRepository;
use App\Repositories\WarehouseTransferRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Abstractions\AmazonInboundShipmentSourceInterface;
use Modules\Amazon\Data\AmazonFbaInboundShipmentData;
use Modules\Amazon\Data\AmazonFbaInboundShipFromMappingData;
use Modules\Amazon\Data\AmazonFbaInboundShipmentWithMappingData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Entities\AmazonFbaInboundShipment;
use Modules\Amazon\Entities\AmazonFbaInboundShipmentItem;
use Modules\Amazon\Entities\AmazonFbaInboundShipFromMapping;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Enums\Entities\FbaInboundShipmentStatusEnum;
use Modules\Amazon\Exceptions\ShipmentStatusNotEligibleForWarehouseTransferCreationException;
use Modules\Amazon\Exceptions\UnmappedAmazonSellerSkuException;
use Modules\Amazon\Exceptions\WarehouseTransferLinesDontMatchInboundException;
use Modules\Amazon\Jobs\AmazonProcessInboundShipmentCancellationsJob;
use Modules\Amazon\Jobs\AmazonProcessUnprocessedInboundShipmentsJob;
use Modules\Amazon\Jobs\CreateAmazonRefreshFbaInboundShipmentItemsJobs;
use Modules\Amazon\Repositories\AmazonFbaInboundShipFromMappingRepository;
use Modules\Amazon\Repositories\AmazonFbaInboundShipmentItemRepository;
use Modules\Amazon\Repositories\AmazonFbaInboundShipmentRepository;
use Modules\Amazon\Repositories\AmazonFnskuRepository;
use Modules\Amazon\Repositories\AmazonProductRepository;
use Modules\Amazon\Repositories\AmazonReportRepository;
use Modules\Amazon\Services\AmazonClient;
use Spatie\LaravelData\DataCollection;
use Spatie\LaravelData\Optional;
use Throwable;

/*
 * TODO: Purchase order may exist prior to inbound shipment but warehouse transfer would not
 *  since for MVP we are allowing user to create the inbound directly on Seller Central, the User needs to initiate
 *  a purchase order to get the shipment going.  It would be great to be able to have a purchase order template email
 *  specific to fba inbound process.  So actually purchase orders could arguably be created after inbound creation
 *  and seller central acts like the data entry point.  This way the User knows it is a confirmed Amazon inbound shipment
 *  before sending details to the supplier
 *
 * TODO: In relation to inventory suggestion reports, once an inbound is imported, it would lower the calculated amounts
 *  the report says are needed to send in (i.e. the resource would have to pull in existing inbounds in progress for a
 *  given sku)
 */

class AmazonInboundManager extends AbstractSalesChannelManager
{
    protected AmazonFbaInboundShipmentRepository $shipments;

    protected WarehouseTransferRepository $warehouseTransfers;

    protected WarehouseTransferManager $transferProcessor;

    protected AmazonReportRepository $amazonReportRepository;

    protected AmazonFnskuRepository $fnskus;

    protected AmazonFbaInboundShipmentItemRepository $shipmentItems;

    protected PurchaseOrderRepository $purchaseOrders;

    protected AmazonProductRepository $amazonProducts;

    protected AmazonFbaInboundShipFromMappingRepository $shipFromMappings;

    /**
     * @throws Exception
     */
    public function __construct(protected AmazonIntegrationInstance $amazonIntegrationInstance)
    {
        parent::__construct($amazonIntegrationInstance, new AmazonClient($amazonIntegrationInstance));
        $this->shipments               = app(AmazonFbaInboundShipmentRepository::class);
        $this->shipmentItems           = app(AmazonFbaInboundShipmentItemRepository::class);
        $this->amazonReportRepository  = app(AmazonReportRepository::class);
        $this->fnskus  = app(AmazonFnskuRepository::class);
        $this->warehouseTransfers      = app(WarehouseTransferRepository::class);
        $this->transferProcessor       = app(WarehouseTransferManager::class);
        $this->purchaseOrders = app(PurchaseOrderRepository::class);
        $this->amazonProducts          = app(AmazonProductRepository::class);
        $this->shipFromMappings        = app(AmazonFbaInboundShipFromMappingRepository::class);
    }


    protected function fetchShipment(string $shipmentId): ClientResponseDataInterface
    {
        return $this->client->getFbaInboundShipments(null, [$shipmentId]);
    }

    public function getShipment(string $shipmentId): ?AmazonFbaInboundShipment
    {
        $shipmentCollection = $this->fetchShipment($shipmentId)->collection;

        if (!$shipmentCollection->count()) {
            return null;
        }

        $this->shipments->saveForIntegration($this->amazonIntegrationInstance, $shipmentCollection->toCollection());

        $this->getShipmentItems($shipmentId);

        return $this->shipments->getFromShipmentId($this->amazonIntegrationInstance, $shipmentId);
    }

    /**
     * @throws Exception
     */
    public function refreshShipments(?string $lastUpdatedAfter = null, ?string $nextToken = null): AmazonResponseData
    {
        customlog('amazon', 'Refreshing Inbound Shipments');
        /*
         * TODO: Imperfect system here.  Can end up skipping a lot of records if:
         *  Processing takes longer than a minute
         *  Processing fails to download everything up to current date (common if there are more than 50 results)
         *      May help if there is pagination enabled
         */

        $lastUpdatedAfter = !empty($lastUpdatedAfter) ? $lastUpdatedAfter :
            (
            !empty($dbLastUpdated = $this->shipments->getLastUpdateDate()) ?
                Carbon::parse($dbLastUpdated)->subMinute()->format('Y-m-d H:i:s') :
                // This could be too much data, going to use fba inventory start date for now
                //Helpers::setting(Setting::KEY_INVENTORY_START_DATE)->format("Y-m-d H:i:s")
                Carbon::parse($this->amazonIntegrationInstance->fbaInventoryTrackingStartDate())
            );

        $amazonResponseDto = $this->client->getFbaInboundShipments(
            null,
            null,
            $lastUpdatedAfter,
            null,
            $nextToken
        );

        $amazonFbaInboundShipmentCollection = $amazonResponseDto->collection;

        if ($duplicates = $this->shipments->duplicatedByNewInbounds(
            $this->amazonIntegrationInstance,
            $amazonFbaInboundShipmentCollection->toCollection()->pluck('json_object.ShipmentId')
        )) {
            $amazonFbaInboundShipmentCollection = $amazonFbaInboundShipmentCollection->reject(function (AmazonFbaInboundShipmentData $shipment) use ($duplicates) {
                return $duplicates->contains($shipment->json_object['ShipmentId']);
            });
        };

        if (!$amazonFbaInboundShipmentCollection->count()) {
            return $amazonResponseDto;
        }

        $this->shipments
            ->saveForIntegration($this->amazonIntegrationInstance, $amazonFbaInboundShipmentCollection->toCollection());

        $amazonFbaInboundShipFromMappingCollection = AmazonFbaInboundShipFromMappingData::collection($amazonFbaInboundShipmentCollection->map(
            function (AmazonFbaInboundShipmentData $inboundShipmentData) {
                return AmazonFbaInboundShipFromMappingData::from([
                    'integration_instance_id' => $this->amazonIntegrationInstance->id,
                    'name' => $inboundShipmentData->json_object['ShipFromAddress']['Name'],
                ]);
            }
        ));

        app(AbstractRepository::class)->save($amazonFbaInboundShipFromMappingCollection->toCollection(),
            AmazonFbaInboundShipFromMapping::class);

        CreateAmazonRefreshFbaInboundShipmentItemsJobs::dispatch($this->amazonIntegrationInstance);

        AmazonProcessUnprocessedInboundShipmentsJob::dispatch($this->amazonIntegrationInstance);
        AmazonProcessInboundShipmentCancellationsJob::dispatch($this->amazonIntegrationInstance);

        return $amazonResponseDto;
    }

    public function refreshShipmentItems(array $shipmentIds): void
    {
        foreach ($shipmentIds as $shipmentId) {
            $this->getShipmentItems($shipmentId);
        }
    }

    public function getShipmentItems(string $shipmentId): AmazonResponseData
    {
        $shipmentItems = $this->fetchShipmentItems($shipmentId);

        $shipment = $this->shipments->getFromShipmentId($this->amazonIntegrationInstance, $shipmentId);

        $this->shipmentItems->save($shipment->id, $shipmentItems->collection);

        return $shipmentItems;
    }

    protected function fetchShipmentItems(string $shipmentId): AmazonResponseData
    {
        return $this->client->getFbaInboundShipmentItems($shipmentId);
    }

    /**
     * @throws Throwable
     */
    public function linkPurchaseOrder(PurchaseOrder $purchaseOrder, string $inboundShipmentId): null|PurchaseOrder|AmazonInboundShipmentSourceInterface
    {
        $shipment = $this->createOrLinkPurchaseOrder(AmazonFbaInboundShipmentWithMappingData::from([
            'shipment' => $this->shipments->getFromShipmentId($this->amazonIntegrationInstance, $inboundShipmentId),
            'mapping' => AmazonFbaInboundShipFromMapping::whereName($purchaseOrder->supplier->name)->first(),
        ]));

        return $shipment->skuLink;
    }

    public function processCancellations(): void
    {
        $this->shipments->getUnprocessedCancellations($this->amazonIntegrationInstance)->each(function (AmazonFbaInboundShipment $shipment) {
            $shipment->skuLink->delete();
        });
    }

    public function processUnprocessed(): void
    {
        $settings = $this->amazonIntegrationInstance->integration_settings;
        $types = [
            'automatically_create_warehouse_transfers_from_inbounds' => Warehouse::class,
            'automatically_create_purchase_orders_from_inbounds' => Supplier::class,
        ];

        $idsToProcess = collect($types)
            ->filter(function ($type, $setting) use ($settings) {
                return $settings[$setting] ?? false;
            })
            ->flatMap(function ($type) {
                return $this->shipments->getUnprocessedWithMappingOfType($this->amazonIntegrationInstance, $type)->pluck('id');
            })
            ->all();

        if (!empty($idsToProcess)) {
            $this->process($idsToProcess);
        }
    }

    /**
     * @throws Throwable
     */
    public function processWarehouseTransfer(int $id, int $warehouse_id): WarehouseTransfer
    {
        $shipment = $this->createOrLinkWarehouseTransfer(AmazonFbaInboundShipmentWithMappingData::from([
            'shipment' => $this->shipments->getForValue($id, 'id', AmazonFbaInboundShipment::class),
            'warehouse' => Warehouse::findOrFail($warehouse_id),
        ]));
        if (!$shipment->sku_link_id)
        {
            throw new Exception($shipment->errorLog);
        }
        return WarehouseTransfer::findOrFail($shipment->sku_link_id);
    }

    public function process(array|int $ids, ?int $warehouse_id = null): Collection
    {
        if (!is_array($ids)) {
            $ids = [$ids];
        }

        $shipments = $this->shipments->getForValues($ids, 'id', AmazonFbaInboundShipment::class);

        $warehouse = $warehouse_id ? Warehouse::findOrFail($warehouse_id) : null;
        $mappings = $this->shipFromMappings->getMappings();

        $filterForLinkType = function ($linkType) use ($warehouse, $mappings) {
            return function ($shipment) use ($warehouse, $mappings, $linkType)
            {
                $data['shipment'] = $shipment;

                if ($warehouse && $linkType == Warehouse::class) {
                    $data['warehouse'] = $warehouse;
                } else {
                    // Early return if no warehouse specified and no mappings available
                    if (!$mapping = $mappings->filter(fn ($item) => strtolower($item->name) == strtolower($shipment->ShipFromName))
                        ->where('link_type', $linkType)
                        ->first()) {
                        return null;
                    }
                    $data['mapping'] = $mapping;
                }

                return AmazonFbaInboundShipmentWithMappingData::from($data);
            };
        };

        $warehouseTransferProcessCollection = $shipments->map($filterForLinkType(Warehouse::class))->filter();
        $purchaseOrderProcessCollection = $shipments->map($filterForLinkType(Supplier::class))->filter();

        $processedShipments = collect();
        $processedShipments->push(...$this->processWarehouseTransfers($warehouseTransferProcessCollection));
        $processedShipments->push(...$this->processPurchaseOrders($purchaseOrderProcessCollection));

        // TODO: Trigger audit trail processing here

        return $processedShipments;
    }

    public function unprocess(array $ids): void
    {
        $shipments = $this->shipments->getForValues($ids, 'id', AmazonFbaInboundShipment::class);

        // Reject any shipments that don't have a link
        $shipments = $shipments->reject(function (AmazonFbaInboundShipment $shipment) {
            return !$shipment->sku_link_id;
        });

        $shipments->each(function (AmazonFbaInboundShipment $shipment) {
            $shipment->skuLink->delete();
            $shipment->skuLink()->dissociate();
            $shipment->save();
        });
    }

    public function processWarehouseTransfers(Collection $warehouseTransferProcessCollection): Collection
    {
        $shipments = collect();
        $warehouseTransferProcessCollection->each(/**
         * @throws Throwable
         */ function (AmazonFbaInboundShipmentWithMappingData $data) use ($shipments) {
             $shipments->push($this->createOrLinkWarehouseTransfer($data));
        });
        return $shipments;
    }

    public function processPurchaseOrders(Collection $purchaseOrderProcessCollection): Collection
    {
        $shipments = collect();
        $purchaseOrderProcessCollection->each(/**
         * @throws Throwable
         */ function (AmazonFbaInboundShipmentWithMappingData $data) use ($shipments) {
            $shipments->push($this->createOrLinkPurchaseOrder($data));
        });
        return $shipments;
    }

    /**
     * @throws Throwable
     */
    public function createOrLinkWarehouseTransfer(AmazonFbaInboundShipmentWithMappingData $data): AmazonFbaInboundShipment {
        $shipment = $data->shipment;

        try {
            /** @var WarehouseTransfer $warehouseTransfer */
            if ($warehouseTransfer = $this->warehouseTransfers->getFromWarehouseTransferNumber($shipment->ShipmentId))
            {
                if ($warehouseTransfer->warehouseTransferLines->isEmpty()) {
                    $this->getWarehouseTransferLines($data, $warehouseTransfer)->each(function (WarehouseTransferLineDto $line) use ($warehouseTransfer) {
                        $this->warehouseTransfers->createWarehouseTransferLine($warehouseTransfer, $line->product_id, $line->quantity);
                    });
                    $warehouseTransfer->save();
                } elseif (!$this->checkLinesMatchInbound($warehouseTransfer, $shipment)) {
                    throw new WarehouseTransferLinesDontMatchInboundException("Warehouse Transfer $warehouseTransfer->warehouse_transfer_number already exists but lines do not match inbound shipment");
                }

                $shipment->skuLink()->associate($warehouseTransfer);
                $shipment->save();

                // TODO: Is this creating movements if lines are added?

                return $shipment;
            } else
            {
                $shipmentStatus = FbaInboundShipmentStatusEnum::ALLOWED_SHIPMENT_STATUS_MAPPINGS[$shipment->ShipmentStatus] ?? null;

                // If Shipment Status wasn't in the enum of allowed values, log the error
                if (!$shipmentStatus) {
                    throw new ShipmentStatusNotEligibleForWarehouseTransferCreationException("Shipment Status $shipment->ShipmentStatus is not a Shipment Status eligible for Warehouse Transfer creation");
                }

                $warehouseTransferDto = WarehouseTransferDto::from([
                    'warehouse_transfer_number' => $shipment->ShipmentId,
                    'transfer_date' => $this->getDateFromShipmentName($shipment->ShipmentName) ?? Carbon::now(),
                    'from_warehouse_id' => ! ($data->warehouse instanceof Optional) ? $data->warehouse->id : $data->mapping->link_id,
                    'to_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
                    // Always start with draft status so that later actions can perform necessary inventory movements
                    'transfer_status' => WarehouseTransfer::TRANSFER_STATUS_DRAFT,
                    //'shipment_status' => WarehouseTransfer::TRANSFER_SHIPMENT_STATUS_UNSHIPPED,
                    //'receipt_status' => WarehouseTransfer::TRANSFER_RECEIPT_STATUS_UNRECEIVED,
                    'warehouseTransferLines' => $this->getWarehouseTransferLines($data),
                ]);

                DB::transaction(function () use ($warehouseTransferDto, $shipment) {
                    $warehouseTransfer = $this->warehouseTransfers->saveWithRelations(WarehouseTransferDto::collection([$warehouseTransferDto]));

                    $this->transferProcessor->openWarehouseTransfer($warehouseTransfer, []);

                    $shipment->skuLink()->associate($warehouseTransfer);
                    $shipment->save();
                });
            }
        } catch (UnmappedAmazonSellerSkuException|WarehouseTransferLinesDontMatchInboundException|ShipmentStatusNotEligibleForWarehouseTransferCreationException $e) {
            /*
             * TODO: InsufficientStockException is not caught right now, but it should be... need to redo how the exception is passed down and now the shipment error log is updated.
             */
            $shipment->errorLog = $e->getMessage();
            $shipment->save();
            return $shipment;
        }

        return $shipment;
    }

    /**
     * @throws Throwable
     */
    public function createOrLinkPurchaseOrder(AmazonFbaInboundShipmentWithMappingData $data
    ): AmazonFbaInboundShipment
    {
        $shipment = $data->shipment;

        try {
            /** @var PurchaseOrder $purchaseOrder */
            if ($purchaseOrder = $this->purchaseOrders->getForValue($shipment->ShipmentId, 'purchase_order_number', PurchaseOrder::class))
            {
                if ($purchaseOrder->purchaseOrderLines->isEmpty()) {
                    $purchaseOrder->setPurchaseOrderLines($this->getPurchaseOrderLines($data, $purchaseOrder)->toArray());
                    $purchaseOrder->save();
                } elseif (!$this->checkLinesMatchInbound($purchaseOrder, $shipment)) {
                    $shipment->errorLog = "Purchase Order $purchaseOrder->purchase_order_number already exists but lines do not match inbound shipment";
                    $shipment->save();
                    return $shipment;
                }

                $shipment->skuLink()->associate($purchaseOrder);
                $shipment->save();

                return $shipment;
            } else
            {

                $shipmentStatus = FbaInboundShipmentStatusEnum::ALLOWED_SHIPMENT_STATUS_MAPPINGS[$shipment->ShipmentStatus] ?? null;

                // If Shipment Status wasn't in the enum of allowed values, log the error
                if (!$shipmentStatus) {
                    $shipment->errorLog = "Shipment Status $shipment->ShipmentStatus is not a Shipment Status eligible for Purchase Order creation";
                    $shipment->save();
                    return $shipment;
                }

                $purchaseOrderDto = PurchaseOrderDto::from([
                    'supplier_id' => $data->mapping->link_id,
                    'purchase_order_number' => $shipment->ShipmentId,
                    'sequence' => PurchaseOrder::getNextSequence(),
                    'purchase_order_date' => $this->getDateFromShipmentName($shipment->ShipmentName) ?? Carbon::now(),
                    'destination_warehouse_id' => $this->amazonIntegrationInstance->warehouse->id,
                    'order_status' => PurchaseOrder::STATUS_OPEN,
                    'currency_id' => Currency::default()->id,
                    'currency_rate' => Currency::default()->conversion,
                    'purchaseOrderLines' => $this->getPurchaseOrderLines($data),
                ]);

                $purchaseOrder = $this->purchaseOrders->saveWithRelations(PurchaseOrderDto::collection([$purchaseOrderDto]));

                $shipment->skuLink()->associate($purchaseOrder);
                $shipment->save();
            }
        } catch (UnmappedAmazonSellerSkuException|SupplierProductNotFoundException $e) {
            $shipment->errorLog = $e->getMessage();
            $shipment->save();
            return $shipment;
        }

        return $shipment;
    }

    public function getDateFromShipmentName($shipmentName): ?Carbon
    {
        // Regular expression to match different date formats in the shipment name
        $pattern = '/(\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2})|(\d{2}\/\d{2}\/\d{4} \d{2}:\d{2})/';

        // Try to find a date in the shipment name
        if (preg_match($pattern, $shipmentName, $matches)) {
            // Filter out empty matches
            $dateString = array_filter($matches)[0];

            // Determine the format of the date string
            $format = !str_contains($dateString, '-') ? 'm/d/Y H:i' : 'Y-m-d H:i:s';

            // Return a Carbon instance
            return Carbon::createFromFormat($format, $dateString, $this->integrationInstance->getTimezone())->setTimezone('UTC');
        }

        return null;
    }

    private function checkLinesMatchInbound(
        AmazonInboundShipmentSourceInterface $model,
        AmazonFbaInboundShipment $shipment
    ): bool {
        $lineCollection = $model->{$model::getLinesRelationName()}->map(function ($line) {
            return IdQuantityData::from([
                'id' => $line->product_id,
                'quantity' => $line->quantity,
            ])->toArray();
        })->toArray();

        $shipmentLineCollection = $shipment->amazonFbaInboundShipmentItems->map(function (AmazonFbaInboundShipmentItem $line)
        {
            $product = $this->getProductFromShipmenItem($line);

            return IdQuantityData::from([
                'id' => $product->id,
                'quantity' => $line->QuantityShipped,
            ])->toArray();
        })->toArray();

        return $lineCollection == $shipmentLineCollection;
    }

    /**
     * @throws UnmappedAmazonSellerSkuException
     */
    private function getProductFromShipmenItem(AmazonFbaInboundShipmentItem $shipmentItem): Product
    {
        if (!$product = $this->fnskus->getProductFromFnsku($this->amazonIntegrationInstance,
            $shipmentItem->FulfillmentNetworkSKU)) {
            $product = $shipmentItem->amazonProduct?->productListing?->product;
        }
        if (!$product) {
            throw new UnmappedAmazonSellerSkuException("$shipmentItem->SellerSKU needs to be mapped before creating this from inbound.  FNSKU: ".$shipmentItem->FulfillmentNetworkSKU.' not initialized either.');
        }
        return $product;
    }

    private function getWarehouseTransferLines(AmazonFbaInboundShipmentWithMappingData $data, ?WarehouseTransfer $warehouseTransfer = null): DataCollection
    {
        $shipment = $data->shipment;
        return WarehouseTransferLineDto::collection($shipment->amazonFbaInboundShipmentItems->map(function (AmazonFbaInboundShipmentItem $shipmentItem) use ($warehouseTransfer)
        {
            if (!$product = $this->fnskus->getProductFromFnsku($this->amazonIntegrationInstance,
                $shipmentItem->FulfillmentNetworkSKU)) {
                $product = $shipmentItem->amazonProduct?->productListing?->product;
            }
            throw_if(
                !$product,
                new UnmappedAmazonSellerSkuException("$shipmentItem->SellerSKU needs to be mapped before creating this warehouse transfer.  FNSKU: ".$shipmentItem->FulfillmentNetworkSKU.' not initialized either.')
            );

            $lineData = [
                'product_id' => $product->id,
                'quantity' => $shipmentItem->QuantityShipped,
            ];

            if ($warehouseTransfer) {
                $lineData['warehouse_transfer_id'] = $warehouseTransfer->id;
            }

            return WarehouseTransferLineDto::from($lineData);
        }));
    }

    private function getPurchaseOrderLines(AmazonFbaInboundShipmentWithMappingData $data, ?PurchaseOrder $purchaseOrder = null): DataCollection
    {
        $shipment = $data->shipment;
        return PurchaseOrderLineDto::collection($shipment->amazonFbaInboundShipmentItems->map(function (AmazonFbaInboundShipmentItem $shipmentItem) use ($data, $purchaseOrder)
        {
            $product = $this->getProductFromShipmenItem($shipmentItem);

            /** @var SupplierProduct $defaultSupplierProduct */
            $defaultSupplierProduct = $product->supplierProducts()->where('supplier_id', $data->mapping->link_id)->first();

            if(!$defaultSupplierProduct) {
                throw new SupplierProductNotFoundException("No default supplier product found for $product->sku for supplier {$data->mapping->link->name}");
            }

            $amount = $defaultSupplierProduct->getDefaultSupplierPricing()?->price ?? $product->unit_cost;

            $lineData = [
                'product_id' => $product->id,
                'quantity' => $shipmentItem->QuantityShipped,
                'amount' => $amount,'received_quantity' => 0,
                'description' => $product->name,
                'nominal_code_id' => $product->cogs_nominal_code_id ?: Helpers::setting(Setting::KEY_NC_MAPPING_COGS),
                'created_at' => Carbon::now()->toDateTimeString(),
                'updated_at' => Carbon::now()->toDateTimeString(),
            ];

            if ($purchaseOrder) {
                $lineData['purchase_order_id'] = $purchaseOrder->id;
            }

            return PurchaseOrderLineDto::from($lineData);
        }));
    }
}
