<?php

namespace Modules\Amazon\Managers;

use App\Data\SalesChannelProductToSkuProductMappingCollectionData;
use App\Data\SalesChannelProductToSkuProductMappingData;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Modules\Amazon\Data\AmazonFnskuGenerationData;
use Modules\Amazon\Data\LinkAmazonFnskuProductToSkuProductData;
use Modules\Amazon\Entities\AmazonFnskuProduct;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonProduct;
use Modules\Amazon\Entities\AmazonReport;
use Modules\Amazon\Enums\Entities\AmazonProductFulfillmentChannelEnum;
use Modules\Amazon\Repositories\AmazonFnskuRepository;
use Modules\Amazon\Repositories\AmazonProductRepository;
use Throwable;

class AmazonFnskuProductManager
{
    private AmazonFnskuRepository $fnskus;
    private AmazonProductRepository $products;

    public function __construct(
        private readonly AmazonIntegrationInstance $integrationInstance
    )
    {
        $this->fnskus = app(AmazonFnskuRepository::class);
        $this->products = app(AmazonProductRepository::class);
    }

    public function retryMappings(): Collection
    {
        $fnskusWithoutProducts = $this->fnskus->fnskusWithoutProducts($this->integrationInstance);

        $fnskuProductCollection = new Collection();
        $fnskusWithoutProducts->each(/**
         * @throws Throwable
         */ function (AmazonFnskuProduct $fnskuProduct) use ($fnskuProductCollection) {
             // Since this is missing the ASIN, it will only try using information in initial inventory
            $fnskuProductCollection->push($this->generateFnskuProduct(new AmazonFnskuGenerationData(
                $fnskuProduct->integration_instance_id,
                $fnskuProduct->fnsku,
                $fnskuProduct->location,
                $fnskuProduct->disposition
            )));
        });

        return $fnskusWithoutProducts;
    }

    /**
     * @throws Throwable
     */
    public function generateFnskuProducts(?AmazonReport $amazonReport = null): int
    {
        customlog('amazon', 'generateFnskuProducts');
        $fnskuGenerationDataCollection = $this->fnskus->getFnskusNeedingGeneration($this->integrationInstance, $amazonReport);
        customlog('amazon', 'generateFnskuProducts generation data: '.count($fnskuGenerationDataCollection));
        foreach ($fnskuGenerationDataCollection as $fnskuGenerationData) {
            customlog('amazon', "\t".'generateFnskuProducts missing generation data', $fnskuGenerationData->toArray());
            (new AmazonFnskuProductManager($this->integrationInstance))->generateFnskuProduct($fnskuGenerationData);
        }
        return count($fnskuGenerationDataCollection);
    }

    public function remapFnskuProducts(SalesChannelProductToSkuProductMappingCollectionData $productMappingDataCollection): void
    {
        $sellerSkusUnmapped = $productMappingDataCollection->mapping->toCollection()->whereNull('mapped_sku')->pluck('salesChannelProduct')->map(function (AmazonProduct $amazonProduct) {
            return $amazonProduct->seller_sku;
        });
        $this->fnskus->getFnskuExtendedData($this->integrationInstance, $sellerSkusUnmapped)->each(function (AmazonFnskuGenerationData $fnskuGenerationData) {
            $fnskuProduct = AmazonFnskuProduct::find($fnskuGenerationData->id);
            $this->linkFnskuProductToSkuProduct(LinkAmazonFnskuProductToSkuProductData::from([
                'id' => $fnskuProduct->id,
                'product_id' => null,
            ]));
        });

        $fnskusForRemapping = $this->fnskus->getFnskuExtendedData($this->integrationInstance, $productMappingDataCollection->mapping->toCollection()->whereNotNull('mapped_sku')->pluck('salesChannelProduct.seller_sku'))->toCollection()->filter();

        $productMappingDataCollection->mapping->toCollection()->whereNotNull('mapped_sku')->each(function (SalesChannelProductToSkuProductMappingData $productMappingData) use ($fnskusForRemapping) {
            /** @var AmazonProduct $amazonProduct */
            $amazonProduct = $productMappingData->salesChannelProduct;
            $fnskusForRemapping->where('msku', $amazonProduct->seller_sku)->each(function (AmazonFnskuGenerationData $fnskuGenerationData) use ($productMappingData) {
                $fnskuProduct = AmazonFnskuProduct::find($fnskuGenerationData->id);
                $this->linkFnskuProductToSkuProduct(LinkAmazonFnskuProductToSkuProductData::from([
                    'id' => $fnskuProduct->id,
                    'product_id' => $productMappingData->product->id,
                ]));
            });
        });
    }

    /**
     * @throws Throwable
     */
    public function generateFnskuProduct(AmazonFnskuGenerationData $fnskuGenerationData): AmazonFnskuProduct
    {
        $amazonProducts = $this->findBestAmazonProductMatchForFnsku($fnskuGenerationData);
        $isSingleProduct = $amazonProducts->count() === 1;
        $productId = $isSingleProduct ? $amazonProducts->first()->id : null;
        $amazonProducts = $amazonProducts->map(function ($amazonProduct) {
            return [
                'id' => $amazonProduct->id,
                'sku' => $amazonProduct->seller_sku,
            ];
        });
        $potentialProductMatches = $isSingleProduct ? null : $amazonProducts->toArray() ?? null;

        return $this->fnskus->saveAmazonFnskuProductFromGeneratedData($this->integrationInstance, $fnskuGenerationData, $productId, $potentialProductMatches);
    }

    public function findBestAmazonProductMatchForFnsku(AmazonFnskuGenerationData $fnskuGenerationData): EloquentCollection
    {
        $amazonProducts = $this->products->findAmazonProductThroughInitialInventory($this->integrationInstance, $fnskuGenerationData);

        if ($amazonProducts->isEmpty() && $fnskuGenerationData->asin) {
            $amazonProducts = $this->products->findAmazonProductsThroughAsin($this->integrationInstance, $fnskuGenerationData);
        }

        if ($amazonProducts->count() === 1 || $amazonProducts->isEmpty()) {
            return $amazonProducts;
        }

        // Reject the least likely matches
        return $amazonProducts->reject(function ($amazonProduct) {
            return $amazonProduct->fulfillment_channel === AmazonProductFulfillmentChannelEnum::DEFAULT;
        });
    }

    public function linkFnskuProductToSkuProduct(LinkAmazonFnskuProductToSkuProductData $data): AmazonFnskuProduct
    {
        return $this->fnskus->linkFnskuProductToSkuProduct($data);
    }
}