<?php

namespace Modules\Amazon\Managers;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelManager;
use App\Data\AccountingTransactionData;
use App\Data\AccountingTransactionLineData;
use App\Helpers;
use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Models\ExternalSourceFinancialLine;
use App\Models\Setting;
use App\Repositories\Accounting\AccountingTransactionRepository;
use App\Repositories\FinancialLineRepository;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Actions\Finance\TransformAmazonFinancialEventToSkuFinancialLines;
use Modules\Amazon\ApiDataTransferObjects\AmazonGetFinancialEventsAdt;
use Modules\Amazon\Data\AmazonFinancialShipmentEventData;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Entities\AmazonFinancialEvent;
use Modules\Amazon\Entities\AmazonFinancialEventGroup;
use Modules\Amazon\Entities\AmazonFinancialRefundEvent;
use Modules\Amazon\Entities\AmazonFinancialShipmentEvent;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Entities\AmazonReportSettlementData;
use Modules\Amazon\Enums\Entities\AmazonFinancialEventGroupAccountingStatusEnum;
use Modules\Amazon\Repositories\AmazonFinancialAdjustmentEventRepository;
use Modules\Amazon\Repositories\AmazonFinancialEventGroupRepository;
use Modules\Amazon\Repositories\AmazonFinancialEventRepository;
use Modules\Amazon\Repositories\AmazonFinancialRefundEventRepository;
use Modules\Amazon\Repositories\AmazonFinancialShipmentEventRepository;
use Modules\Amazon\Repositories\AmazonOrderRepository;
use Modules\Amazon\Services\AmazonClient;
use Throwable;

class AmazonFinanceManager extends AbstractSalesChannelManager
{
    protected AmazonFinancialEventGroupRepository $amazonFinancialEventGroupRepository;

    protected AmazonFinancialEventRepository $amazonFinancialEventRepository;

    protected AmazonFinancialShipmentEventRepository $shipmentEvents;

    protected AmazonFinancialRefundEventRepository $refundEvents;

    protected AmazonFinancialAdjustmentEventRepository $adjustmentEvents;

    protected AccountingTransactionRepository $accountingTransactionRepository;

    protected AmazonOrderRepository $amazonOrders;

    /**
     * @throws Exception
     */
    public function __construct(protected AmazonIntegrationInstance $amazonIntegrationInstance)
    {
        parent::__construct($amazonIntegrationInstance, new AmazonClient($amazonIntegrationInstance));
        $this->amazonFinancialEventGroupRepository = app(AmazonFinancialEventGroupRepository::class);
        $this->amazonFinancialEventRepository = app(AmazonFinancialEventRepository::class);
        $this->accountingTransactionRepository = app(AccountingTransactionRepository::class);
        $this->shipmentEvents = app(AmazonFinancialShipmentEventRepository::class);
        $this->refundEvents = app(AmazonFinancialRefundEventRepository::class);
        $this->adjustmentEvents = app(AmazonFinancialAdjustmentEventRepository::class);
        $this->amazonOrders = app(AmazonOrderRepository::class);
    }

    /**
     * @throws Exception
     */
    public function refreshFinancialEventGroups(?string $startedAfter = null, ?string $nextToken = null): AmazonResponseData
    {
        // TODO: Date range cannot span more than 365 days
        $startedAfter = ! empty($startedAfter) ? $startedAfter :
            (
                ! empty($dbStarted = $this->amazonFinancialEventGroupRepository->getLastStartedDate($this->amazonIntegrationInstance)) ?
                    Carbon::parse($dbStarted)->format('Y-m-d H:i:s') :
                    Helpers::setting(Setting::KEY_INVENTORY_START_DATE)->format('Y-m-d H:i:s')
            );

        if (Carbon::parse($startedAfter)->diffInDays(Carbon::now()) > 365) {
            $startedBefore = Carbon::parse($startedAfter)->addDays(365)->format('Y-m-d H:i:s');
        } else {
            $startedBefore = null;
        }

        $amazonResponseDto = $this->client->getFinancialEventGroups($startedAfter, $startedBefore, $nextToken);

        $amazonFinancialEventGroupCollection = $amazonResponseDto->collection;

        if (! $amazonFinancialEventGroupCollection->count()) {
            return $amazonResponseDto;
        }

        $this->amazonFinancialEventGroupRepository
            ->save($this->amazonIntegrationInstance, $amazonFinancialEventGroupCollection->toCollection());

        return $amazonResponseDto;
    }

    /**
     * @throws Exception
     */
    public function syncFinancialEvents(
        ?AmazonFinancialEventGroup $amazonFinancialEventGroup = null,
        ?string $postedAfter = null,
        ?string $nextToken = null
    ): AmazonResponseData {
        // TODO: Date range cannot span more than 180 days.  Can we handle this?
        $postedAfter = ! empty($postedAfter) ? $postedAfter :
            (
                ! empty($dbStarted = $this->amazonFinancialEventRepository->getLastPostedDate($this->amazonIntegrationInstance)) ?
                    Carbon::parse($dbStarted)->format('Y-m-d H:i:s') :
                    ($amazonFinancialEventGroup ? null : Helpers::setting(Setting::KEY_INVENTORY_START_DATE)->format('Y-m-d H:i:s'))
            );

        $amazonResponseDto = $amazonFinancialEventGroup ? $this->client
            ->getFinancialEventsByGroupId($amazonFinancialEventGroup, $postedAfter, $nextToken) :
            $this->client->getFinancialEvents($postedAfter);

        $amazonFinancialEventCollection = $amazonResponseDto->collection;

        if (! $amazonFinancialEventCollection->count()) {
            return $amazonResponseDto;
        }

        $this->amazonFinancialEventRepository
            ->save($this->amazonIntegrationInstance, $amazonFinancialEventCollection);

        return $amazonResponseDto;
    }

    /**
     * @throws Exception
     */
    public function refreshFinancialEvents(AmazonGetFinancialEventsAdt $parameters): AmazonResponseData
    {
        $amazonResponseDto = $this->client->getFinancialEvents($parameters);

        // Shipment events
        $shipmentEventList = AmazonFinancialShipmentEventData::collection(
            $amazonResponseDto->collection['ShipmentEventList']->json_object
        )->toCollection();
        $this->shipmentEvents->save(
            $this->amazonIntegrationInstance,
            $shipmentEventList
        );

        //Refund Events
        $refundEventList = AmazonFinancialShipmentEventData::collection(
            $amazonResponseDto->collection['RefundEventList']->json_object
        )->toCollection();
        $this->refundEvents->save(
            $this->amazonIntegrationInstance,
            $refundEventList
        );

        //Adjustment Events
        /**
         * Remove records with 0 amounts
         */
        $amazonResponseDto->collection['AdjustmentEventList']->json_object = collect($amazonResponseDto->collection['AdjustmentEventList']->json_object)->map(function (array $object) {
            if (@$object['AdjustmentAmount']['CurrencyAmount'] != 0) {
                return $object;
            }

            return null;
        })
            ->filter()
            ->values()
            ->toArray();

        $adjustmentEventList = AmazonFinancialShipmentEventData::collection(
            $amazonResponseDto->collection['AdjustmentEventList']->json_object
        )->toCollection();
        $this->adjustmentEvents->save(
            $this->amazonIntegrationInstance,
            $adjustmentEventList
        );

        return $amazonResponseDto;
    }

    public function createFinancialLines(): void
    {
        $financialLines = collect();

        $financialLines = $this->mergeFinancialLines($financialLines, $this->refundEvents, AmazonFinancialRefundEvent::class);
        $financialLines = $this->mergeFinancialLines($financialLines, $this->shipmentEvents, AmazonFinancialShipmentEvent::class);

        app(FinancialLineRepository::class)->save($financialLines, ExternalSourceFinancialLine::class);
    }

    private function mergeFinancialLines(&$financialLines, $events, string $eventType): Collection
    {
        /**
         * @uses AbstractAmazonFinancialEventRepository::getEventsWithoutFinancialLines
         */
        $events->getEventsWithoutFinancialLines($this->amazonIntegrationInstance)
            ->each(function (AmazonFinancialEvent $event) use (&$financialLines) {
                $financialLines = $financialLines->merge((new TransformAmazonFinancialEventToSkuFinancialLines($event))->handle());
            });

        return $financialLines;
    }

    /**
     * @throws Exception
     */
    public function createFinancialEvents(AmazonFinancialEvent $amazonFinancialEvent): void
    {
        $this->amazonFinancialEventRepository->createEventsFromFinancialEvent($amazonFinancialEvent);
    }

    public function createAccountingTransactions(array $amazonFinancialEventGroupIds = []): void
    {
        customlog('amazon', 'createAccountingTransactions');
        $groups = $this->amazonFinancialEventGroupRepository->getFinancialEventGroupsNeedingAccountingTransactions(
            $this->amazonIntegrationInstance, $amazonFinancialEventGroupIds
        );
        customlog('amazon', 'createAccountingTransactions for '.$groups->count().' groups');
        $groups->each(/**
         * @throws Throwable
         */ function (AmazonFinancialEventGroup $amazonFinancialEventGroup) {
            customlog('amazon', 'createAccountingTransactions for group '.$amazonFinancialEventGroup->FinancialEventGroupId);
            // Create accounting transaction for financial event group
            $this->createBatchSalesInvoiceAccountingTransaction($amazonFinancialEventGroup);
        });

        customlog('amazon', 'Update accounting statuses');
        $this->updateAccountingStatuses($this->amazonIntegrationInstance, $amazonFinancialEventGroupIds);
    }

    private function checkIfMissingSettlementDataMapping(AmazonFinancialEventGroup $amazonFinancialEventGroup): bool
    {
        return $amazonFinancialEventGroup->amazonReportSettlementData()->whereDoesntHave('amazonReportSettlementTypeMapping')->exists();
    }

    /**
     * @throws Throwable
     */
    private function createBatchSalesInvoiceAccountingTransaction(AmazonFinancialEventGroup $amazonFinancialEventGroup): void
    {
        DB::transaction(function () use ($amazonFinancialEventGroup) {
            if ($this->checkIfMissingSettlementDataMapping($amazonFinancialEventGroup)) {
                throw new Exception('Missing settlement data mapping for group '.$amazonFinancialEventGroup->FinancialEventGroupId);
            }

            $amazonFinancialEventGroup->amazonReportSettlementData()->each(function (AmazonReportSettlementData $settlementData) use (&$lines) {
                $financialCode = $settlementData->amazonReportSettlementTypeMapping->financial_code;

                $lines[$financialCode] = $lines[$financialCode] ?? [
                    'amount' => 0,
                    'nominal_code_id' => $settlementData->amazonReportSettlementTypeMapping->nominal_code_id,
                ];

                $lines[$financialCode]['amount'] += floatval($settlementData->amount);
            });

            $accountingTransactionLines = collect();

            $total = 0;
            $i = 1;
            foreach ($lines as $description => $line) {
                $accountingTransactionLines->push(AccountingTransactionLineData::from([
                    'link_id' => $i++,
                    'link_type' => 'summarized',
                    'description' => $description,
                    'type' => AccountingTransactionLine::TYPE_SALES_INVOICE_LINE,
                    'quantity' => 1,
                    'amount' => $line['amount'],
                    'nominal_code_id' => $line['nominal_code_id'],
                ]));
                $total += $line['amount'];
            }

            if (round($total, 2) != round($amazonFinancialEventGroup->OriginalTotalCurrencyAmount, 2)) {
                throw new Exception('Total '.$total.' does not match settlement data total '.$amazonFinancialEventGroup->OriginalTotalCurrencyAmount);
            }

            $accountingTransactionCollection = AccountingTransactionData::collection([AccountingTransactionData::from([
                'type' => AccountingTransaction::TYPE_BATCH_SALES_ORDER_INVOICE,
                'name' => 'Amazon',
                'transaction_date' => $amazonFinancialEventGroup->FinancialEventGroupEnd_datetime,
                'reference' => $amazonFinancialEventGroup->FinancialEventGroupId,
                'currency_code' => $amazonFinancialEventGroup->OriginalTotalCurrencyCode,
                'currency_rate' => $amazonFinancialEventGroup->OriginalTotalCurrencyAmount == 0 ? 0 : $amazonFinancialEventGroup->ConvertedTotalCurrencyAmount / $amazonFinancialEventGroup->OriginalTotalCurrencyAmount,
                'link_id' => $amazonFinancialEventGroup->id,
                'link_type' => AmazonFinancialEventGroup::class,
                'accounting_status' => AmazonFinancialEventGroupAccountingStatusEnum::STATUS_DONE,
                'total' => $total,
                'accounting_transaction_lines' => $accountingTransactionLines,
            ])]);

            $accountingTransaction = $this->accountingTransactionRepository->saveWithRelations($accountingTransactionCollection)->first();

            $this->amazonFinancialEventGroupRepository->completeAccountingTransactionLink($amazonFinancialEventGroup, $accountingTransaction);
        });
    }

    private function updateAccountingStatuses(AmazonIntegrationInstance $amazonIntegrationInstance, array $amazonFinancialEventGroupIds): void
    {
        $this->amazonFinancialEventGroupRepository->updateAccountingStatusForGroupsMissingSettlementDataSalesOrderInvoice($amazonIntegrationInstance, $amazonFinancialEventGroupIds);
        $this->amazonFinancialEventGroupRepository->updateAccountingStatusForGroupsMissingSettlementDataSalesOrder($amazonIntegrationInstance, $amazonFinancialEventGroupIds);
        $this->amazonFinancialEventGroupRepository->updateAccountingStatusForGroupsMissingSettlementDataAmazonOrder($amazonIntegrationInstance, $amazonFinancialEventGroupIds);
        $this->amazonFinancialEventGroupRepository->updateAccountingStatusForGroupsMissingSettlementDataMapping($amazonIntegrationInstance, $amazonFinancialEventGroupIds);
        $this->amazonFinancialEventGroupRepository->updateAccountingStatusForGroupsMissingSettlementData($amazonIntegrationInstance, $amazonFinancialEventGroupIds);
    }

    public function seedTypeMappings(): void
    {
        $this->amazonFinancialEventGroupRepository->seedTypeMappings($this->amazonIntegrationInstance);
    }
}
