<?php

namespace Modules\Amazon\Managers;

use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelManager;
use App\DTO\BulkImportResponseDto;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Repositories\SalesOrderFulfillmentRepository;
use Modules\Amazon\Actions\RequestAmazonFeed;
use Modules\Amazon\Entities\AmazonFeedSubmission;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Enums\Entities\AmazonFeedTypeEnum;
use Modules\Amazon\Enums\Entities\FeedSubmissionResultCodeEnum;
use Modules\Amazon\Services\AmazonClient;
use Throwable;

class AmazonFeedManager extends AbstractSalesChannelManager
{
    private SalesOrderFulfillmentRepository $salesOrderFulfillmentRepository;

    public function __construct(protected AmazonIntegrationInstance $amazonIntegrationInstance)
    {
        parent::__construct($amazonIntegrationInstance, new AmazonClient($amazonIntegrationInstance));

        $this->salesOrderFulfillmentRepository = app(SalesOrderFulfillmentRepository::class);
    }

    /**
     * Get and update feed details.
     */
    public function getAmazonFeedDetails(AmazonFeedSubmission $amazonFeedSubmission): AmazonFeedSubmission
    {
        $amazonFeedSubmission->fill($this->client->getFeed($amazonFeedSubmission->feedId)->toArray());

        $amazonFeedSubmission->update();

        // We need to reset sales order fulfillment as non submitted
        if ($amazonFeedSubmission->feedType === AmazonFeedTypeEnum::ORDER_FULFILLMENT && $amazonFeedSubmission->processingStatus === 'ERROR') {
            $amazonFeedSubmission->salesOrderFulfillments->each(function ($salesOrderFulfillment) {
                $this->salesOrderFulfillmentRepository->markAsNotSubmittedToSalesChannel($salesOrderFulfillment);
            });
        }

        return $amazonFeedSubmission;
    }

    /**
     * Get and update feed result document url.
     */
    public function getAmazonFeedResultDocument(AmazonFeedSubmission $amazonFeedSubmission): AmazonFeedSubmission
    {
        $amazonFeedSubmission->url = $this->client->getResultFeedDocument($amazonFeedSubmission->resultFeedDocumentId);
        $amazonFeedSubmission->update();

        $amazonFeedSubmission->resultFeedDocumentXml = $this->client->getResultFeedDocumentXml($amazonFeedSubmission->url);
        $amazonFeedSubmission->update();

        $this->handleAmazonFeedResponse($amazonFeedSubmission);

        return $amazonFeedSubmission;
    }

    public function handleAmazonFeedResponse(AmazonFeedSubmission $amazonFeedSubmission)
    {
        if (! is_null($amazonFeedSubmission->resultFeedDocumentXml)) {
            switch ($amazonFeedSubmission->feedType) {
                case AmazonFeedTypeEnum::ORDER_FULFILLMENT:
                    if ($amazonFeedSubmission->resultFeedDocumentXml) {
                        $resultAmazonMessages = @xmlToArray($amazonFeedSubmission->resultFeedDocumentXml)['Message']['ProcessingReport']['Result'];

                        $requestMessages = collect(xmlToArray($amazonFeedSubmission->requestFeedDocumentXml)['Message']);

                        collect($resultAmazonMessages)->each(function ($amazonResultMessage) use ($amazonFeedSubmission, $requestMessages) {
                            if (is_array($amazonResultMessage) && @$amazonResultMessage['ResultCode'] === FeedSubmissionResultCodeEnum::ERROR->value) {
                                /*
                                * TODO: Additional info won't exist here if there is an error most likely.  To know the amazon order id you have to use the message id and
                                *  correspond it to the feed that was sent.
                                */

                                if (@$amazonResultMessage['MessageID']) {
                                    $requestMessage = ($requestMessages->where('MessageID', $amazonResultMessage['MessageID']))->first();

                                    if ($amazonOrderId = @$requestMessage['OrderFulfillment']['AmazonOrderID']) {
                                        $amazonFeedSubmission->salesOrderFulfillments->each(function (SalesOrderFulfillment $salesOrderFulfillment) use ($amazonOrderId) {
                                            if ($salesOrderFulfillment->salesOrder->sales_order_number === $amazonOrderId) {
                                                $this->salesOrderFulfillmentRepository->markAsNotSubmittedToSalesChannel($salesOrderFulfillment);
                                            }
                                        });
                                    }
                                }
                            }
                        });
                    }

                    break;

                default:
                    // code...
                    break;
            }
        }
    }

    /**
     * Create order acknowledgement feed.
     *
     * @throws Throwable
     */
    public function createOrderAcknowledgementFeedRequest(BulkImportResponseDto $saveSalesOrderResult): void
    {
        $salesOrderLines = SalesOrderLine::query()
            ->selectRaw('sales_orders.sales_order_number as AmazonOrderId')
            ->selectRaw('sales_orders.id as MerchantOrderID')
            ->selectRaw("'Success' as StatusCode")
            ->selectRaw('sales_channel_line_id as Item_AmazonOrderItemCode')
            ->selectRaw('sales_order_lines.id as Item_MerchantOrderItemID')
            ->leftJoinRelationship('salesOrder')
            ->whereIn('sales_orders.sales_order_number', function ($query) use ($saveSalesOrderResult) {
                $query->select('sales_order_number')->from($saveSalesOrderResult->temp_table_name);
            })
            ->get();

        (new RequestAmazonFeed(
            $this->amazonIntegrationInstance->id,
            AmazonFeedTypeEnum::ORDER_ACKNOWLEDGEMENT,
            [],
            $salesOrderLines->toArray()
        ))->handle(
            createImmediately: false,
            checkIfSimilarRequestExists: false
        );
    }
}
