<?php

namespace Modules\Amazon\Jobs;

use Exception;
use Illuminate\Bus\Batchable;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Amazon\Managers\AmazonOrderManager;
use Throwable;

class GenerateCreateSkuOrdersJobs implements ShouldBeUnique, ShouldQueue
{
    use Batchable, Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public string $uniqueId;

    public int $uniqueFor = 60 * 10;

    public int $timeout = 120;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(public AmazonIntegrationInstance $amazonIntegrationInstance)
    {
        $this->onQueue('salesOrderProcessing');
        $this->uniqueId = get_class($this).'-'.$this->amazonIntegrationInstance->id;
    }

    public function tags(): array
    {
        return [
            get_class($this),
            $this->uniqueId,
        ];
    }

    /**
     * Execute the job.
     *
     * @throws Exception
     * @throws Throwable
     */
    public function handle(): void
    {
        customlog('salesOrderProcessing', 'GenerateCreateSkuOrdersJobs');
        (new AmazonOrderManager($this->amazonIntegrationInstance))->getSalesChannelOrdersQuery([], true)->chunk(1000, function ($orders) {
            dispatch(new CreateSkuOrdersJob($this->amazonIntegrationInstance, $orders->pluck('id')->toArray()))->onQueue('salesOrderProcessing');
        });
    }
}
