<?php

namespace Modules\Amazon\Jobs\Abstractions;

use App\Abstractions\Integrations\ClientResponseDataInterface;
use App\Exceptions\ApiException;
use App\Models\Concerns\HasJobPayload;
use DateTime;
use Doctrine\DBAL\Exception\ConnectionException;
use Exception;
use GuzzleHttp\Exception\ConnectException;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use InvalidArgumentException;
use Modules\Amazon\Data\AmazonResponseData;
use Modules\Amazon\Exceptions\AmazonCouldNotResolveHostException;
use Modules\Amazon\Exceptions\AmazonTemporaryBadRequestException;
use Modules\Amazon\Exceptions\AmazonTimeoutException;

/**
 * @method release(int $delay)
 * @method fail(ApiException $ae)
 *
 * @property $job
 */
abstract class AmazonJob implements ShouldQueue
{
    use Dispatchable;
    use HasJobPayload;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    public ?array $jobPayload = [];

    public int $uniqueFor = 15 * 60;

    public int $timeout = 15 * 60;

    public int $maxExceptions = 1;

    abstract public function apiCall(): AmazonResponseData|ClientResponseDataInterface;

    protected function handleApiException(ApiException $ae, int $delay = 60): void
    {
        if ($ae->getCode() == 429) { // Too many requests
            Log::debug('Job '.$this->job->resolveName().': Failed with too many requests, delaying '.$this->job->uuid().' for '.$delay.' seconds');
            $this->release($delay);
        } else {
            $this->fail($ae);
        }
    }

    protected function handleCouldNotResolveHostException(int $delay = 60): void
    {
        $this->release($delay);
    }

    protected function handleTimeoutException(int $delay = 60): void
    {
        $this->release($delay);
    }

    /**
     * Initialization steps for every job.
     *
     * @throws InvalidArgumentException
     */
    protected function jobStarted(): void
    {
        $job = $this->getJobPayload();

        if ($job) {
            $this->jobPayload = $job->payload;
        }
    }

    /**
     * @throws InvalidArgumentException
     */
    protected function jobCompleted(): void
    {
        $this->deleteJobPayload();
    }

    protected function nextPage(): void
    {
        $this->release(0);
    }

    /**
     * Set the value of timeout
     *
     * @return  self
     */
    public function setTimeout($timeout)
    {
        $this->timeout = $timeout;

        $this->uniqueFor = $this->timeout;

        return $this;
    }

    /**
     * Determine the time at which the job should time out.
     */
    public function retryUntil(): DateTime
    {
        return now()->addSeconds($this->timeout);
    }

    /**
     * Execute the job.
     *
     * @throws Exception
     */
    public function handle(): void
    {
        //Get job payload from database.
        $this->jobStarted();

        //Make API call.
        try {
            $amazonResponseDto = $this->apiCall();
        } catch (ApiException $ae) {
            $this->handleApiException($ae);

            return;
        } catch (AmazonCouldNotResolveHostException $e) {
            $this->handleCouldNotResolveHostException();

            return;
        } catch (AmazonTimeoutException|ConnectException|ConnectionException|AmazonTemporaryBadRequestException $e) {
            $this->handleTimeoutException();

            return;
        }

        //Save payload in jobs
        $this->saveJobPayload([
            'nextToken' => $amazonResponseDto->nextToken,
        ]);

        //Get next page
        if ($amazonResponseDto->nextToken) {
            //Log::debug('Job: Next page '.$amazonResponseDto->nextToken);
            $this->nextPage();
        } else {
            $this->jobCompleted();
        }
    }
}
